/* ************************************************************************
*   File: interpreter.c                                   EmpireMUD 2.0b5 *
*  Usage: parse user commands, search for specials, call ACMD functions   *
*                                                                         *
*  EmpireMUD code base by Paul Clarke, (C) 2000-2024                      *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  EmpireMUD based upon CircleMUD 3.0, bpl 17, by Jeremy Elson.           *
*  CircleMUD (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "db.h"
#include "handler.h"
#include "olc.h"
#include "skills.h"
#include "vnums.h"
#include "dg_scripts.h"
#include "constants.h"

/**
* Contents:
*   Command Prototypes
*   Main Command List
*   Command Interpreter
*   Alias System
*   Helper Functions
*   Command Functions
*   Character Creation
*   Menu Interpreter Functions
*/

// prool globals
int prool_flag=0;

// locals
void set_creation_state(descriptor_data *d, int state);
void show_bonus_trait_menu(char_data *ch);
void send_login_motd(descriptor_data *desc, int bad_pws);

 //////////////////////////////////////////////////////////////////////////////
//// COMMAND PROTOTYPES //////////////////////////////////////////////////////

ACMD(do_abandon);
ACMD(do_ability);
ACMD(do_accept);
ACMD(do_addnotes);
ACMD(do_admin_util);
ACMD(do_advance);
ACMD(do_adventure);
ACMD(do_affects);
ACMD(do_alias);
ACMD(do_alternate);
ACMD(do_approach);
ACMD(do_approve);
ACMD(do_assist);
ACMD(do_at);
ACMD(do_automessage);
ACMD(do_autostore);
ACMD(do_autowiz);
ACMD(do_avoid);

ACMD(do_ban);
ACMD(do_barde);
ACMD(do_beckon);
ACMD(do_bite);
ACMD(do_board);
ACMD(do_boost);
ACMD(do_breakreply);
ACMD(do_buffs);
ACMD(do_buildcheck);
ACMD(do_burn);
ACMD(do_buy);
ACMD(do_butcher);

ACMD(do_cast);
ACMD(do_cede);
ACMD(do_changepass);
ACMD(do_chart);
ACMD(do_chip);
ACMD(do_chop);
ACMD(do_circle);
ACMD(do_city);
ACMD(do_claim);
ACMD(do_class);
ACMD(do_clearabilities);
ACMD(do_clearmeters);
ACMD(do_climb);
ACMD(do_coins);
ACMD(do_combine);
ACMD(do_compare);
ACMD(do_command);
ACMD(do_commands);
ACMD(do_companions);
ACMD(do_confer);
ACMD(do_config);
ACMD(do_confirm);
ACMD(do_conjure);
ACMD(do_consider);
ACMD(do_contents);
ACMD(do_cooldowns);
ACMD(do_create);
ACMD(do_credits);
ACMD(do_customize);

ACMD(do_date);
ACMD(do_dc);
ACMD(do_dedicate);
ACMD(do_defect);
ACMD(do_demote);
ACMD(do_deposit);
ACMD(do_designate);
ACMD(do_diagnose);
ACMD(do_diplomacy);
ACMD(do_disembark);
ACMD(do_disguise);
ACMD(do_dismantle);
ACMD(do_dismiss);
ACMD(do_dismount);
ACMD(do_dispatch);
ACMD(do_display);
ACMD(do_distance);
ACMD(do_douse);
ACMD(do_drag);
ACMD(do_drive);
ACMD(do_draw);
ACMD(do_drink);
ACMD(do_drop);

ACMD(do_earthmeld);
ACMD(do_eat);
ACMD(do_echo);
ACMD(do_edelete);
ACMD(do_editnotes);
ACMD(do_eedit);
ACMD(do_efind);
ACMD(do_elog);
ACMD(do_emotd);
ACMD(do_empire_inventory);
ACMD(do_empires);
ACMD(do_empty);
ACMD(do_endwar);
ACMD(do_enroll);
ACMD(do_enter);
ACMD(do_equipment);
ACMD(do_esay);
ACMD(do_estats);
ACMD(do_events);
ACMD(do_examine);
ACMD(do_excavate);
ACMD(do_exchange);
ACMD(do_execute);
ACMD(do_exit);
ACMD(do_exits);
ACMD(do_expel);

ACMD(do_factions);
ACMD(do_feed);
ACMD(do_fightmessages);
ACMD(do_file);
ACMD(do_fillin);
ACMD(do_findmaintenance);
ACMD(do_finish);
ACMD(do_fire);
ACMD(do_flee);
ACMD(do_follow);
ACMD(do_force);
ACMD(do_forgive);
ACMD(do_friend);
ACMD(do_fullsave);

ACMD(do_gecho);
ACMD(do_gen_augment);
ACMD(do_gen_craft);
ACMD(do_gen_door);
ACMD(do_gen_interact_room);
ACMD(do_gen_ps);
ACMD(do_gen_text_string);
ACMD(do_gen_write);
ACMD(do_get);
ACMD(do_give);
ACMD(do_goto);
ACMD(do_grab);
ACMD(do_group);
ACMD(do_gsay);

ACMD(do_harness);
ACMD(do_harvest);
ACMD(do_help);
ACMD(do_helpsearch);
ACMD(do_herd);
ACMD(do_history);
ACMD(do_hit);
ACMD(do_home);
ACMD(do_hostile);
ACMD(do_hunt);

ACMD(do_identify);
ACMD(do_ignore);
ACMD(do_import);
ACMD(do_infiltrate);
ACMD(do_informative);
ACMD(do_instance);
ACMD(do_inspire);
ACMD(do_insult);
ACMD(do_interlink);
ACMD(do_inventory);
ACMD(do_invis);
ACMD(do_island);
ACMD(do_islands);

ACMD(do_keep);

ACMD(do_land);
ACMD(do_last);
ACMD(do_lastname);
ACMD(do_lay);
ACMD(do_lead);
ACMD(do_learn);
ACMD(do_learned);
ACMD(do_library);
ACMD(do_light);
ACMD(do_list);
ACMD(do_load);
ACMD(do_load_vehicle);
ACMD(do_look);
ACMD(do_lore);

ACMD(do_mail);
ACMD(do_maintain);
ACMD(do_manage);
ACMD(do_map);
ACMD(do_mapscan);
ACMD(do_mapsize);
ACMD(do_mark);
ACMD(do_messages);
ACMD(do_meters);
ACMD(do_milk);
ACMD(do_mine);
ACMD(do_minipets);
ACMD(do_mint);
ACMD(do_mirrorimage);
ACMD(do_missing_help_files);
ACMD(do_morph);
ACMD(do_mount);
ACMD(do_move);
ACMD(do_moveeinv);
ACMD(do_mudstats);
ACMD(do_mydescription);

ACMD(do_nearby);
ACMD(do_no_cmd);
ACMD(do_nodismantle);
ACMD(do_noskill);

ACMD(do_offenses);
ACMD(do_olc);
ACMD(do_oset);
ACMD(do_order);

ACMD(do_page);
ACMD(do_paint);
ACMD(do_passives);
ACMD(do_peace);
ACMD(do_pickpocket);
ACMD(do_plant);
ACMD(do_play);
ACMD(do_playerdelete);
ACMD(do_pledge);
ACMD(do_point);
ACMD(do_poofset);
ACMD(do_portal);
ACMD(do_pour);
ACMD(do_progress);
ACMD(do_promote);
ACMD(do_prompt);
ACMD(do_prospect);
ACMD(do_pub_comm);
ACMD(do_publicize);
ACMD(do_purge);
ACMD(do_put);

ACMD(do_quaff);
ACMD(do_quest);
ACMD(do_quit);

ACMD(do_random);
ACMD(do_read);
ACMD(do_ready);
ACMD(do_reboot);
ACMD(do_recipes);
ACMD(do_reclaim);
ACMD(do_recolor);
ACMD(do_reload);
ACMD(do_remove);
ACMD(do_reply);
ACMD(do_rescale);
ACMD(do_respawn);
ACMD(do_respond);
ACMD(do_rest);
ACMD(do_restore);
ACMD(do_retrieve);
ACMD(do_return);
ACMD(do_rework);
ACMD(do_roadsign);
ACMD(do_role);
ACMD(do_roll);
ACMD(do_roster);
ACMD(do_run);

ACMD(do_sacrifice);
ACMD(do_save);
ACMD(do_saw);
ACMD(do_say);
ACMD(do_scan);
ACMD(do_score);
ACMD(do_scrap);
ACMD(do_scrape);
ACMD(do_search);
ACMD(do_seed);
ACMD(do_send);
ACMD(do_selfdelete);
ACMD(do_separate);
ACMD(do_set);
ACMD(do_share);
ACMD(do_shear);
ACMD(do_sheathe);
ACMD(do_ship);
ACMD(do_shoot);
ACMD(do_show);
ACMD(do_sire);
ACMD(do_sit);
ACMD(do_skills);
ACMD(do_skin);
ACMD(do_slash_channel);
ACMD(do_slay);
ACMD(do_sleep);
ACMD(do_snoop);
ACMD(do_socials);
ACMD(do_speak);
ACMD(do_spec_comm);
ACMD(do_specialize);
ACMD(do_split);
ACMD(do_stake);
ACMD(do_stand);
ACMD(do_start);
ACMD(do_stat);
ACMD(do_steal);
ACMD(do_stop);
ACMD(do_store);
ACMD(do_string_editor);
ACMD(do_struggle);
ACMD(do_summary);
ACMD(do_summon);
ACMD(do_survey);
ACMD(do_swap);
ACMD(do_swim);
ACMD(do_switch);
ACMD(do_syslog);

ACMD(do_tame);
ACMD(do_tan);
ACMD(do_tedit);
ACMD(do_tell);
ACMD(do_temperature);
ACMD(do_territory);
ACMD(do_throw);
ACMD(do_tie);
ACMD(do_time);
ACMD(do_tip);
ACMD(do_title);
ACMD(do_toggle);
ACMD(do_tomb);
ACMD(do_track);
ACMD(do_trade);
ACMD(do_trans);
ACMD(do_transport);
ACMD(do_tunnel);

ACMD(do_unban);
ACMD(do_unbind);
ACMD(do_unfriend);
ACMD(do_unharness);
ACMD(do_unload_vehicle);
ACMD(do_unpublicize);
ACMD(do_unquest);
ACMD(do_unpaint);
ACMD(do_unprogress);
ACMD(do_unshare);
ACMD(do_upgrade);
ACMD(do_use);
ACMD(do_users);

ACMD(do_visible);
ACMD(do_vnum);
ACMD(do_vstat);

ACMD(do_wake);
ACMD(do_warehouse);
ACMD(do_wear);
ACMD(do_weather);
ACMD(do_where);
ACMD(do_whereami);
ACMD(do_who);
ACMD(do_whoami);
ACMD(do_whois);
ACMD(do_wield);
ACMD(do_withdraw);
ACMD(do_wizlock);
ACMD(do_wizutil);
ACMD(do_workforce);
ACMD(do_write);

/* DG Script ACMD's */
ACMD(do_tattach);
ACMD(do_tdetach);
ACMD(do_madventurecomplete);
ACMD(do_maggro);
ACMD(do_masound);
ACMD(do_mkill);
ACMD(do_mjunk);
ACMD(do_mdoor);
ACMD(do_mechoaround);
ACMD(do_mechoneither);
ACMD(do_msend);
ACMD(do_mecho);
ACMD(do_mload);
ACMD(do_mlog);
ACMD(do_mmod);
ACMD(do_mmorph);
ACMD(do_mmove);
ACMD(do_mpurge);
ACMD(do_mquest);
ACMD(do_mgoto);
ACMD(do_maoe);
ACMD(do_mat);
ACMD(do_mbuild);
ACMD(do_mdamage);
ACMD(do_mdot);
ACMD(do_mrestore);
ACMD(do_msiege);
ACMD(do_mslay);
ACMD(do_mteleport);
ACMD(do_mterracrop);
ACMD(do_mterraform);
ACMD(do_mforce);
ACMD(do_mheal);
ACMD(do_mhunt);
ACMD(do_mremember);
ACMD(do_mforget);
ACMD(do_mscale);
ACMD(do_mtransform);
ACMD(do_mbuildingecho);
ACMD(do_mown);
ACMD(do_mregionecho);
ACMD(do_msubecho);
ACMD(do_mvehicleecho);
ACMD(do_vdelete);
ACMD(do_mfollow);


 //////////////////////////////////////////////////////////////////////////////
//// MAIN COMMAND LIST ///////////////////////////////////////////////////////

/* This is the Main Command List(tm).
 * You can put new commands in, take commands out, change the order
 * they appear in, etc.  You can adjust the "priority" of commands
 * simply by changing the order they appear in the command list.
 * (For example, if you want "as" to mean "assist" instead of "ask",
 * just put "assist" above "ask" in the Main Command List(tm).
 *
 * In general, utility commands such as "at" should have high priority;
 * infrequently used and dangerously destructive commands should have low
 * priority.
 */
#define NO_MIN  0

#define ABILITY_CMD( string, pos, func, lvl, type, abil )  { string, pos, func, lvl, NO_GRANTS, NO_SCMD, type, NOBITS, abil }
#define GRANT_CMD( string, pos, func, lvl, type, grant )  { string, pos, func, lvl, grant, NO_SCMD, type, NOBITS, NO_ABIL }
#define SIMPLE_CMD( string, pos, func, lvl, type )  { string, pos, func, lvl, NO_GRANTS, NO_SCMD, type, NOBITS, NO_ABIL }
#define SCMD_CMD( string, pos, func, lvl, type, subcmd )  { string, pos, func, lvl, NO_GRANTS, subcmd, type, NOBITS, NO_ABIL }
#define STANDARD_CMD( string, pos, func, lvl, grants, subcmd, type, flags, abil )  { string, pos, func, lvl, grants, subcmd, type, flags, abil }

cpp_extern const struct command_info cmd_info[] = {
	/* this must be first -- for specprocs */
	STANDARD_CMD( "RESERVED", POS_DEAD, NULL, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_UTIL, CMD_NO_ABBREV, NO_ABIL ),
	
	// basic command setup:
	// STANDARD_CMD( "command", POS_x, do_function, MIN_LEVEL, GRANTS_x, SCMD_x, CTYPE_x, CMD_x, ABIL_x ),
	// STANDARD_CMD( "defaults", POS_STANDING, do_function, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_UTIL, NOBITS, NO_ABIL ),

	/* directions come before other commands to preserve abbrevs */
	SCMD_CMD( "north", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, NORTH ),
	SCMD_CMD( "east", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, EAST ),
	SCMD_CMD( "south", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, SOUTH ),
	SCMD_CMD( "west", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, WEST ),
	SCMD_CMD( "northeast", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, NORTHEAST ),
	SCMD_CMD( "northwest", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, NORTHWEST ),
	SCMD_CMD( "southeast", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, SOUTHEAST ),
	SCMD_CMD( "southwest", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, SOUTHWEST ),
	SCMD_CMD( "ne", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, NORTHEAST ),
	SCMD_CMD( "nw", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, NORTHWEST ),
	SCMD_CMD( "se", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, SOUTHEAST ),
	SCMD_CMD( "sw", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, SOUTHWEST ),
	SCMD_CMD( "up", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, UP ),
	SCMD_CMD( "down", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, DOWN ),
	SCMD_CMD( "fore", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, FORE ),
	SCMD_CMD( "starboard", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, STARBOARD ),
	SCMD_CMD( "port", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, PORT ),
	SCMD_CMD( "aft", POS_STANDING, do_move, NO_MIN, CTYPE_MOVE, AFT ),

	/* now, the main list */
	SIMPLE_CMD( "/", POS_DEAD, do_slash_channel, NO_MIN, CTYPE_COMM ),

	// . is "olc" for imms but "bookedit" for mortals
	GRANT_CMD( ".", POS_DEAD, do_olc, LVL_BUILDER, CTYPE_OLC, GRANT_OLC ),
	SCMD_CMD( ".", POS_STANDING, do_library, NO_MIN, CTYPE_UTIL, SCMD_BOOKEDIT ),
	
	SIMPLE_CMD( "at", POS_DEAD, do_at, LVL_START_IMM, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "abandon", POS_RESTING, do_abandon, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "ability", POS_DEAD, do_ability, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "abilities", POS_DEAD, do_ability, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "accept", POS_DEAD, do_accept, NO_MIN, CTYPE_UTIL, SCMD_ACCEPT ),
	SIMPLE_CMD( "adventure", POS_RESTING, do_adventure, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "addnotes", POS_STANDING, do_addnotes, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_EDITNOTES ),
	GRANT_CMD( "advance", POS_DEAD, do_advance, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_ADVANCE ),
	SCMD_CMD( "alias", POS_DEAD, do_alias, NO_MIN, CTYPE_UTIL, SCMD_ALIAS ),
	SIMPLE_CMD( "alternate", POS_DEAD, do_alternate, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "affects", POS_DEAD, do_affects, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "approach", POS_FIGHTING, do_approach, NO_MIN, CTYPE_COMBAT ),
	STANDARD_CMD( "approve", POS_DEAD, do_approve, LVL_CIMPL, GRANT_APPROVE, SCMD_APPROVE, CTYPE_IMMORTAL, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "assist", POS_FIGHTING, do_assist, NO_MIN, CTYPE_COMBAT ),
	SCMD_CMD( "ask", POS_RESTING, do_spec_comm, NO_MIN, CTYPE_COMM, SCMD_ASK ),
	SIMPLE_CMD( "autostore", POS_RESTING, do_autostore, LVL_CIMPL, CTYPE_IMMORTAL ),
	GRANT_CMD( "automessage", POS_DEAD, do_automessage, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_AUTOMESSAGE),
	GRANT_CMD( "automsg", POS_DEAD, do_automessage, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_AUTOMESSAGE),
	SIMPLE_CMD( "autowiz", POS_DEAD, do_autowiz, LVL_CIMPL, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "avoid", POS_STANDING, do_avoid, NO_MIN, CTYPE_MOVE ),

	STANDARD_CMD( "build", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_BUILD, CTYPE_BUILD, NOBITS, NO_ABIL ),
	STANDARD_CMD( "bake", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_BAKE, CTYPE_BUILD, NOBITS, NO_ABIL ),
	GRANT_CMD( "ban", POS_DEAD, do_ban, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_BAN ),
	STANDARD_CMD( "barde", POS_STANDING, do_barde, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_SKILL, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "beckon", POS_RESTING, do_beckon, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "bite", POS_STUNNED, do_bite, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMBAT, CMD_NO_ANIMALS, NO_ABIL ),
	SCMD_CMD( "board", POS_STANDING, do_board, NO_MIN, CTYPE_MOVE, SCMD_BOARD ),
	ABILITY_CMD( "boost", POS_RESTING, do_boost, NO_MIN, CTYPE_UTIL, ABIL_BOOST ),
	SCMD_CMD( "bookedit", POS_STANDING, do_library, NO_MIN, CTYPE_UTIL, SCMD_BOOKEDIT ),
	STANDARD_CMD( "brew", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_BREW, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "breakreply", POS_DEAD, do_breakreply, LVL_START_IMM, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "buffs", POS_SLEEPING, do_buffs, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "buildcheck", POS_SLEEPING, do_buildcheck, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "buy", POS_RESTING, do_buy, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "bug", POS_DEAD, do_gen_write, NO_MIN, CTYPE_COMM, SCMD_BUG ),
	SCMD_CMD( "burn", POS_SITTING, do_burn, NO_MIN, CTYPE_UTIL, SCMD_BURN ),
	SIMPLE_CMD( "butcher", POS_STANDING, do_butcher, NO_MIN, CTYPE_SKILL ),
	SCMD_CMD( "brief", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_TOGGLE ),

	STANDARD_CMD( "chop", POS_STANDING, do_chop, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SCMD_CMD( "cast", POS_SLEEPING, do_cast, NO_MIN, CTYPE_SKILL, SCMD_CAST ),
	SIMPLE_CMD( "cd", POS_DEAD, do_cooldowns, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "cede", POS_DEAD, do_cede, NO_MIN, CTYPE_EMPIRE ),
	SCMD_CMD( "chant", POS_SLEEPING, do_cast, NO_MIN, CTYPE_SKILL, SCMD_CHANT ),
	SIMPLE_CMD( "chart", POS_DEAD, do_chart, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "changepass", POS_DEAD, do_changepass, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "chip", POS_STANDING, do_chip, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "circle", POS_STANDING, do_circle, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "cities", POS_DEAD, do_city, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "city", POS_DEAD, do_city, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "claim", POS_RESTING, do_claim, NO_MIN, CTYPE_EMPIRE ),
	// uncomment this if you use the old class system
	// SIMPLE_CMD( "class", POS_DEAD, do_class, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "clear", POS_DEAD, do_gen_ps, NO_MIN, CTYPE_UTIL, SCMD_CLEAR ),
	SIMPLE_CMD( "clearmeters", POS_DEAD, do_clearmeters, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "clearabilities", POS_DEAD, do_clearabilities, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_CLEARABILITIES ),
	SIMPLE_CMD( "climb", POS_STANDING, do_climb, NO_MIN, CTYPE_MOVE ),
	SCMD_CMD( "close", POS_SITTING, do_gen_door, NO_MIN, CTYPE_MOVE, SCMD_CLOSE ),
	SCMD_CMD( "cls", POS_DEAD, do_gen_ps, NO_MIN, CTYPE_UTIL, SCMD_CLEAR ),
	SCMD_CMD( "coins", POS_DEAD, do_coins, NO_MIN, CTYPE_UTIL, TRUE ),
	SIMPLE_CMD( "combine", POS_RESTING, do_combine, NO_MIN, CTYPE_UTIL ),
	ABILITY_CMD( "command", POS_STANDING, do_command, NO_MIN, CTYPE_SKILL, ABIL_VAMP_COMMAND ),
	SCMD_CMD( "commands", POS_DEAD, do_commands, NO_MIN, CTYPE_UTIL, SCMD_COMMANDS ),
	SIMPLE_CMD( "companions", POS_DEAD, do_companions, NO_MIN, CTYPE_SKILL ),
	SIMPLE_CMD( "compare", POS_RESTING, do_compare, NO_MIN, CTYPE_SKILL ),
	SIMPLE_CMD( "consider", POS_RESTING, do_consider, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "config", POS_DEAD, do_config, LVL_CIMPL, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "confirm", POS_DEAD, do_confirm, NO_MIN, CTYPE_UTIL ),
	ABILITY_CMD( "confer", POS_RESTING, do_confer, NO_MIN, CTYPE_SKILL, ABIL_CONFER ),
	SIMPLE_CMD( "conjure", POS_DEAD, do_conjure, NO_MIN, CTYPE_SKILL ),
	SIMPLE_CMD( "contents", POS_RESTING, do_contents, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "cook", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_COOK, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "cooldowns", POS_DEAD, do_cooldowns, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "craft", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_CRAFT, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SCMD_CMD( "credits", POS_DEAD, do_gen_text_string, NO_MIN, CTYPE_UTIL, TEXT_FILE_CREDITS ),
	SIMPLE_CMD( "create", POS_STANDING, do_create, LVL_GOD, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "customize", POS_STANDING, do_customize, NO_MIN, CTYPE_BUILD ),
	SCMD_CMD( "compact", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_TOGGLE ),

	SCMD_CMD( "date", POS_DEAD, do_date, LVL_START_IMM, CTYPE_IMMORTAL, SCMD_DATE ),
	GRANT_CMD( "dc", POS_DEAD, do_dc, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_DC ),
	SCMD_CMD( "drink", POS_RESTING, do_drink, NO_MIN, CTYPE_UTIL, SCMD_DRINK ),
	SIMPLE_CMD( "dedicate", POS_STANDING, do_dedicate, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "demote", POS_DEAD, do_demote, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "deposit", POS_STANDING, do_deposit, NO_MIN, CTYPE_EMPIRE ),
	STANDARD_CMD( "defect", POS_DEAD, do_defect, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_EMPIRE, CMD_NO_ABBREV, NO_ABIL ),
	SCMD_CMD( "designate", POS_STANDING, do_designate, NO_MIN, CTYPE_BUILD, SCMD_DESIGNATE ),
	SIMPLE_CMD( "diagnose", POS_RESTING, do_diagnose, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "diplomacy", POS_DEAD, do_diplomacy, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "directions", POS_DEAD, do_distance, NO_MIN, CTYPE_IMMORTAL ),
	ABILITY_CMD( "disguise", POS_STANDING, do_disguise, NO_MIN, CTYPE_SKILL, ABIL_DISGUISE ),
	SIMPLE_CMD( "dismount", POS_SITTING, do_dismount, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "dismantle", POS_STANDING, do_dismantle, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "dismiss", POS_STANDING, do_dismiss, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "disembark", POS_STANDING, do_disembark, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "distance", POS_DEAD, do_distance, NO_MIN, CTYPE_IMMORTAL ),
	STANDARD_CMD( "dig", POS_STANDING, do_gen_interact_room, NO_MIN, NO_GRANTS, INTERACT_DIG, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "display", POS_DEAD, do_display, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "dispatch", POS_RESTING, do_dispatch, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "douse", POS_STANDING, do_douse, NO_MIN, CTYPE_BUILD ),
	SCMD_CMD( "drop", POS_RESTING, do_drop, NO_MIN, CTYPE_MOVE, SCMD_DROP ),
	STANDARD_CMD( "drag", POS_STANDING, do_drag, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "draw", POS_RESTING, do_draw, NO_MIN, CTYPE_COMBAT ),
	STANDARD_CMD( "drive", POS_SITTING, do_drive, NO_MIN, NO_GRANTS, SCMD_DRIVE, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),

	SCMD_CMD( "eat", POS_RESTING, do_eat, NO_MIN, CTYPE_UTIL, SCMD_EAT ),
	ABILITY_CMD( "earthmeld", POS_STUNNED, do_earthmeld, NO_MIN, CTYPE_MOVE, ABIL_EARTHMELD ),
	STANDARD_CMD( "echo", POS_SLEEPING, do_echo, LVL_CIMPL, GRANT_ECHO, SCMD_ECHO, CTYPE_IMMORTAL, NOBITS, NO_ABIL ),
	GRANT_CMD( "editnotes", POS_STANDING, do_editnotes, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_EDITNOTES ),
	SIMPLE_CMD( "eedit", POS_DEAD, do_eedit, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "efind", POS_SLEEPING, do_efind, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "elog", POS_DEAD, do_elog, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "empires", POS_DEAD, do_empires, NO_MIN, CTYPE_EMPIRE ),
	SCMD_CMD( "empireidentify", POS_DEAD, do_empire_inventory, NO_MIN, CTYPE_EMPIRE, SCMD_EIDENTIFY ),
	SCMD_CMD( "empireinventory", POS_DEAD, do_empire_inventory, NO_MIN, CTYPE_EMPIRE, SCMD_EINVENTORY ),
	SIMPLE_CMD( "empty", POS_STANDING, do_empty, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "einventory", POS_DEAD, do_empire_inventory, NO_MIN, CTYPE_EMPIRE, SCMD_EINVENTORY ),
	SCMD_CMD( "eidentify", POS_DEAD, do_empire_inventory, NO_MIN, CTYPE_EMPIRE, SCMD_EIDENTIFY ),
	SIMPLE_CMD( "emotd", POS_DEAD, do_emotd, NO_MIN, CTYPE_EMPIRE ),
	SCMD_CMD( "emote", POS_RESTING, do_echo, NO_MIN, CTYPE_COMM, SCMD_EMOTE ),
	SCMD_CMD( ":", POS_RESTING, do_echo, NO_MIN, CTYPE_COMM, SCMD_EMOTE ),
	SCMD_CMD( "empirehistory", POS_DEAD, do_history, NO_MIN, CTYPE_COMM, SCMD_EMPIRE_HISTORY ),
	SCMD_CMD( "ehistory", POS_DEAD, do_history, NO_MIN, CTYPE_COMM, SCMD_EMPIRE_HISTORY ),
	STANDARD_CMD( "enchant", POS_STANDING, do_gen_augment, NO_MIN, NO_GRANTS, AUGMENT_ENCHANTMENT, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	GRANT_CMD( "endwar", POS_DEAD, do_endwar, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_EMPIRES ),
	SIMPLE_CMD( "enter", POS_STANDING, do_enter, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "enroll", POS_DEAD, do_enroll, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "equipment", POS_DEAD, do_equipment, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "esay", POS_DEAD, do_esay, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "etalk", POS_DEAD, do_esay, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "estats", POS_DEAD, do_estats, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "events", POS_DEAD, do_events, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "examine", POS_SITTING, do_examine, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "excavate", POS_STANDING, do_excavate, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "exchange", POS_STANDING, do_exchange, NO_MIN, CTYPE_BUILD ),
	STANDARD_CMD( "execute", POS_STANDING, do_execute, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMBAT, CMD_WHILE_FEEDING, NO_ABIL ),
	SIMPLE_CMD( "exit", POS_RESTING, do_exit, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "exits", POS_RESTING, do_exits, NO_MIN, CTYPE_UTIL, -1 ),
	SCMD_CMD( "export", POS_DEAD, do_import, NO_MIN, CTYPE_UTIL, TRADE_EXPORT ),
	SIMPLE_CMD( "expel", POS_DEAD, do_expel, NO_MIN, CTYPE_EMPIRE ),
	STANDARD_CMD( "edelete", POS_DEAD, do_edelete, LVL_CIMPL, GRANT_EMPIRES, NO_SCMD, CTYPE_EMPIRE, CMD_NO_ABBREV, NO_ABIL ),

	SCMD_CMD( "fill", POS_RESTING, do_pour, NO_MIN, CTYPE_UTIL, SCMD_FILL ),
	SIMPLE_CMD( "factions", POS_DEAD, do_factions, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "fastmorph", POS_RESTING, do_morph, NO_MIN, CTYPE_MOVE, SCMD_FASTMORPH ),
	SIMPLE_CMD( "feed", POS_STANDING, do_feed, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "fightmessages", POS_DEAD, do_fightmessages, NO_MIN, CTYPE_UTIL, SCMD_FIGHT ),
	SCMD_CMD( "fmessages", POS_DEAD, do_fightmessages, NO_MIN, CTYPE_UTIL, SCMD_FIGHT ),
	SIMPLE_CMD( "file", POS_DEAD, do_file, LVL_START_IMM, CTYPE_IMMORTAL ),
	STANDARD_CMD( "fillin", POS_STANDING, do_fillin, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "finish", POS_DEAD, do_finish, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "findmaintenance", POS_DEAD, do_findmaintenance, NO_MIN, CTYPE_EMPIRE ),
	STANDARD_CMD( "fire", POS_SITTING, do_fire, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMBAT, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "fish", POS_SITTING, do_gen_interact_room, NO_MIN, NO_GRANTS, INTERACT_FISH, CTYPE_SKILL, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "flee", POS_FIGHTING, do_flee, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMBAT, CMD_NO_ABBREV, NO_ABIL ),
	SIMPLE_CMD( "follow", POS_RESTING, do_follow, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "forage", POS_SITTING, do_gen_interact_room, NO_MIN, NO_GRANTS, INTERACT_FORAGE, CTYPE_SKILL, CMD_NO_ANIMALS, NO_ABIL ),
	GRANT_CMD( "force", POS_SLEEPING, do_force, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_FORCE ),
	STANDARD_CMD( "forge", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_FORGE, CTYPE_BUILD, NOBITS, NO_ABIL ),
	GRANT_CMD( "forgive", POS_DEAD, do_forgive, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_FORGIVE ),
	SCMD_CMD( "fprompt", POS_DEAD, do_prompt, NO_MIN, CTYPE_UTIL, SCMD_FPROMPT ),
	SIMPLE_CMD( "friends", POS_DEAD, do_friend, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "fullsave", POS_DEAD, do_fullsave, LVL_TOP, CTYPE_IMMORTAL ),
	STANDARD_CMD( "freeze", POS_DEAD, do_wizutil, LVL_CIMPL, GRANT_FREEZE, SCMD_FREEZE, CTYPE_IMMORTAL, NOBITS, NO_ABIL ),

	SIMPLE_CMD( "get", POS_RESTING, do_get, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "gather", POS_SITTING, do_gen_interact_room, NO_MIN, NO_GRANTS, INTERACT_GATHER, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	GRANT_CMD( "gecho", POS_DEAD, do_gecho, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_GECHO ),
	SIMPLE_CMD( "give", POS_RESTING, do_give, NO_MIN, CTYPE_MOVE ),
	SCMD_CMD( "goto", POS_SLEEPING, do_goto, LVL_START_IMM, CTYPE_IMMORTAL, SCMD_GOTO ),
	SCMD_CMD( "godnet", POS_DEAD, do_pub_comm, LVL_GOD, CTYPE_IMMORTAL, SCMD_GODNET ),
	SCMD_CMD( "godhistory", POS_DEAD, do_history, LVL_GOD, CTYPE_COMM, SCMD_GOD_HISTORY ),
	SCMD_CMD( "ghistory", POS_DEAD, do_history, LVL_GOD, CTYPE_COMM, SCMD_GOD_HISTORY ),
	SCMD_CMD( "godlist", POS_DEAD, do_gen_text_string, NO_MIN, CTYPE_UTIL, TEXT_FILE_GODLIST ),
	SCMD_CMD( "gold", POS_DEAD, do_coins, NO_MIN, CTYPE_UTIL, TRUE ),
	SIMPLE_CMD( "grab", POS_RESTING, do_grab, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "group", POS_DEAD, do_group, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "gsay", POS_DEAD, do_gsay, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "gtell", POS_DEAD, do_gsay, NO_MIN, CTYPE_COMM ),
	SCMD_CMD( "gossip", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_GOSSIP ),

	SIMPLE_CMD( "help", POS_DEAD, do_help, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "handbook", POS_DEAD, do_gen_text_string, LVL_START_IMM, CTYPE_IMMORTAL, TEXT_FILE_HANDBOOK ),
	STANDARD_CMD( "harness", POS_STANDING, do_harness, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "harvest", POS_STANDING, do_harvest, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "herd", POS_STANDING, do_herd, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "helpsearch", POS_DEAD, do_helpsearch, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "hint", POS_DEAD, do_tip, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "history", POS_DEAD, do_history, NO_MIN, CTYPE_COMM, SCMD_HISTORY ),
	STANDARD_CMD( "hit", POS_FIGHTING, do_hit, NO_MIN, NO_GRANTS, SCMD_HIT, CTYPE_COMBAT, CMD_WHILE_FEEDING, NO_ABIL ),
	SIMPLE_CMD( "hold", POS_RESTING, do_grab, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "home", POS_DEAD, do_home, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "hone", POS_STANDING, do_gen_augment, NO_MIN, NO_GRANTS, AUGMENT_HONE, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	GRANT_CMD( "hostile", POS_DEAD, do_hostile, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_HOSTILE ),
	SIMPLE_CMD( "hunt", POS_STANDING, do_hunt, NO_MIN, CTYPE_SKILL ),

	SIMPLE_CMD( "inventory", POS_DEAD, do_inventory, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "identify", POS_RESTING, do_identify, NO_MIN, CTYPE_SKILL ),
	SCMD_CMD( "idea", POS_DEAD, do_gen_write, NO_MIN, CTYPE_COMM, SCMD_IDEA ),
	SIMPLE_CMD( "ignore", POS_DEAD, do_ignore, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "import", POS_DEAD, do_import, NO_MIN, CTYPE_UTIL, TRADE_IMPORT ),
	SCMD_CMD( "imotd", POS_DEAD, do_gen_text_string, LVL_START_IMM, CTYPE_IMMORTAL, TEXT_FILE_IMOTD ),
	SIMPLE_CMD( "infiltrate", POS_STANDING, do_infiltrate, NO_MIN, CTYPE_MOVE ),
	SCMD_CMD( "info", POS_DEAD, do_gen_text_string, NO_MIN, CTYPE_UTIL, TEXT_FILE_INFO ),
	SIMPLE_CMD( "informative", POS_DEAD, do_informative, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "instance", POS_DEAD, do_instance, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_INSTANCE ),
	ABILITY_CMD( "inspire", POS_STANDING, do_inspire, NO_MIN, CTYPE_SKILL, ABIL_INSPIRE ),
	SIMPLE_CMD( "insult", POS_RESTING, do_insult, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "interlink", POS_STANDING, do_interlink, NO_MIN, CTYPE_BUILD ),
	SIMPLE_CMD( "invis", POS_DEAD, do_invis, LVL_GOD, CTYPE_IMMORTAL ),
	GRANT_CMD( "island", POS_DEAD, do_island, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_ISLAND ),
	SIMPLE_CMD( "islands", POS_DEAD, do_islands, NO_MIN, CTYPE_EMPIRE ),

	STANDARD_CMD( "junk", POS_RESTING, do_drop, NO_MIN, NO_GRANTS, SCMD_JUNK, CTYPE_UTIL, CMD_NO_ABBREV, NO_ABIL ),

	STANDARD_CMD( "kill", POS_FIGHTING, do_hit, NO_MIN, NO_GRANTS, SCMD_KILL, CTYPE_COMBAT, CMD_WHILE_FEEDING, NO_ABIL ),
	SCMD_CMD( "keep", POS_DEAD, do_keep, NO_MIN, CTYPE_UTIL, SCMD_KEEP ),

	SCMD_CMD( "look", POS_RESTING, do_look, NO_MIN, CTYPE_UTIL, SCMD_LOOK ),
	STANDARD_CMD( "lay", POS_STANDING, do_lay, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "land", POS_FIGHTING, do_land, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "last", POS_DEAD, do_last, LVL_START_IMM, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "lastname", POS_DEAD, do_lastname, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "lead", POS_STANDING, do_lead, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "learn", POS_STANDING, do_learn, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "learned", POS_DEAD, do_learned, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "leave", POS_STANDING, do_exit, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "light", POS_SITTING, do_light, NO_MIN, CTYPE_UTIL, SCMD_LIGHT ),
	SIMPLE_CMD( "list", POS_SITTING, do_list, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "library", POS_STANDING, do_library, NO_MIN, CTYPE_UTIL, SCMD_LIBRARY ),
	GRANT_CMD( "load", POS_DEAD, do_load, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_LOAD ),
	STANDARD_CMD( "load", POS_STANDING, do_load_vehicle, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "loadvehicle", POS_STANDING, do_load_vehicle, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	GRANT_CMD( "lore", POS_DEAD, do_lore, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_LORE ),
	SCMD_CMD( "levels", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_LEVELS ),

	SIMPLE_CMD( "mount", POS_STANDING, do_mount, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "maintain", POS_STANDING, do_maintain, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "make", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_MAKE, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "manage", POS_DEAD, do_manage, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "manufacture", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_MANUFACTURE, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "map", POS_RESTING, do_map, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "mapscan", POS_RESTING, do_mapscan, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "mapsize", POS_DEAD, do_mapsize, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "mark", POS_RESTING, do_mark, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "meters", POS_DEAD, do_meters, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "melt", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_SMELT, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "messages", POS_DEAD, do_messages, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "mine", POS_STANDING, do_mine, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "minipets", POS_DEAD, do_minipets, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "mill", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_MILL, CTYPE_BUILD, NOBITS, NO_ABIL ),
	STANDARD_CMD( "milk", POS_STANDING, do_milk, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "mint", POS_STANDING, do_mint, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	ABILITY_CMD( "mirrorimage", POS_FIGHTING, do_mirrorimage, NO_MIN, CTYPE_COMBAT, ABIL_MIRRORIMAGE ),
	SIMPLE_CMD( "missinghelp", POS_DEAD, do_missing_help_files, LVL_START_IMM, CTYPE_IMMORTAL ),
	STANDARD_CMD( "mix", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_MIX, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SCMD_CMD( "morph", POS_FIGHTING, do_morph, NO_MIN, CTYPE_MOVE, SCMD_MORPH ),
	SCMD_CMD( "motd", POS_DEAD, do_gen_text_string, NO_MIN, CTYPE_UTIL, TEXT_FILE_MOTD ),
	GRANT_CMD( "moveeinv", POS_DEAD, do_moveeinv, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_EMPIRES ),
	SIMPLE_CMD( "mudstats", POS_DEAD, do_mudstats, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "murder", POS_FIGHTING, do_hit, NO_MIN, NO_GRANTS, SCMD_KILL, CTYPE_COMBAT, CMD_WHILE_FEEDING, NO_ABIL ),
	SIMPLE_CMD( "mail", POS_STANDING, do_mail, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "mute", POS_DEAD, do_wizutil, LVL_CIMPL, GRANT_MUTE, SCMD_MUTE, CTYPE_IMMORTAL, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "mydescription", POS_STANDING, do_mydescription, NO_MIN, CTYPE_UTIL ),

	SIMPLE_CMD( "nearby", POS_RESTING, do_nearby, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "news", POS_DEAD, do_gen_text_string, NO_MIN, CTYPE_UTIL, TEXT_FILE_NEWS ),
	SIMPLE_CMD( "nodismantle", POS_SLEEPING, do_nodismantle, NO_MIN, CTYPE_BUILD ),
	SIMPLE_CMD( "noskill", POS_DEAD, do_noskill, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "notitle", POS_DEAD, do_wizutil, LVL_CIMPL, CTYPE_IMMORTAL, SCMD_NOTITLE ),

	SIMPLE_CMD( "order", POS_RESTING, do_order, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "offenses", POS_DEAD, do_offenses, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "oset", POS_DEAD, do_oset, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_OSET ),
	SCMD_CMD( "open", POS_SITTING, do_gen_door, NO_MIN, CTYPE_MOVE, SCMD_OPEN ),
	SCMD_CMD( "oocsay", POS_RESTING, do_say, NO_MIN, CTYPE_COMM, SCMD_OOCSAY ),
	SCMD_CMD( "osay", POS_RESTING, do_say, NO_MIN, CTYPE_COMM, SCMD_OOCSAY ),
	SCMD_CMD( "\"", POS_RESTING, do_say, NO_MIN, CTYPE_COMM, SCMD_OOCSAY ),
	GRANT_CMD( "olc", POS_DEAD, do_olc, LVL_BUILDER, CTYPE_OLC, GRANT_OLC ),

	SIMPLE_CMD( "put", POS_RESTING, do_put, NO_MIN, CTYPE_MOVE ),
	GRANT_CMD( "page", POS_DEAD, do_page, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_PAGE ),
	SIMPLE_CMD( "paint", POS_STANDING, do_paint, NO_MIN, CTYPE_BUILD ),
	STANDARD_CMD( "pan", POS_STANDING, do_gen_interact_room, NO_MIN, NO_GRANTS, INTERACT_PAN, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "party", POS_DEAD, do_group, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "passives", POS_DEAD, do_passives, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "peace", POS_DEAD, do_peace, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_PEACE ),
	STANDARD_CMD( "pick", POS_SITTING, do_gen_interact_room, NO_MIN, NO_GRANTS, INTERACT_PICK, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "pickpocket", POS_STANDING, do_pickpocket, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMBAT, CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "pilot", POS_SITTING, do_drive, NO_MIN, NO_GRANTS, SCMD_PILOT, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "play", POS_STANDING, do_play, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "playerdelete", POS_SLEEPING, do_playerdelete, LVL_CIMPL, GRANT_PLAYERDELETE, NO_SCMD, CTYPE_IMMORTAL, CMD_NO_ABBREV, NO_ABIL ),
	STANDARD_CMD( "plant", POS_STANDING, do_plant, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "pledge", POS_SLEEPING, do_pledge, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "point", POS_RESTING, do_point, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "policy", POS_DEAD, do_gen_text_string, NO_MIN, CTYPE_UTIL, TEXT_FILE_POLICY ),
	SCMD_CMD( "poofin", POS_DEAD, do_poofset, LVL_GOD, CTYPE_IMMORTAL, SCMD_POOFIN ),
	SCMD_CMD( "poofout", POS_DEAD, do_poofset, LVL_GOD, CTYPE_IMMORTAL, SCMD_POOFOUT ),
	SIMPLE_CMD( "portal", POS_STANDING, do_portal, NO_MIN, CTYPE_MOVE ),
	SCMD_CMD( "pour", POS_RESTING, do_pour, NO_MIN, CTYPE_UTIL, SCMD_POUR ),
	STANDARD_CMD( "press", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_PRESS, CTYPE_BUILD, NOBITS, NO_ABIL ),
	STANDARD_CMD( "process", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_PROCESS, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "progress", POS_DEAD, do_progress, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "promote", POS_DEAD, do_promote, NO_MIN, CTYPE_EMPIRE ),
	SCMD_CMD( "prompt", POS_DEAD, do_prompt, NO_MIN, CTYPE_UTIL, SCMD_PROMPT ),
	STANDARD_CMD( "prospect", POS_STANDING, do_prospect, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "publicize", POS_RESTING, do_publicize, NO_MIN, CTYPE_EMPIRE ),
	GRANT_CMD( "purge", POS_DEAD, do_purge, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_PURGE ),
	SIMPLE_CMD( "psay", POS_DEAD, do_gsay, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "ptell", POS_DEAD, do_gsay, NO_MIN, CTYPE_COMM ),
	SCMD_CMD( "practice", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_PRACTICE ),

	SIMPLE_CMD( "quests", POS_DEAD, do_quest, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "quaff", POS_RESTING, do_quaff, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "quarry", POS_STANDING, do_gen_interact_room, NO_MIN, NO_GRANTS, INTERACT_QUARRY, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "quit", POS_DEAD, do_quit, NO_MIN, NO_GRANTS, SCMD_QUIT, CTYPE_UTIL, CMD_NO_ABBREV, NO_ABIL ),

	SIMPLE_CMD( "reply", POS_DEAD, do_reply, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "random", POS_SLEEPING, do_random, LVL_START_IMM, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "read", POS_SITTING, do_read, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "ready", POS_RESTING, do_ready, NO_MIN, CTYPE_COMBAT ),
	STANDARD_CMD( "reboot", POS_DEAD, do_reboot, LVL_CIMPL, GRANT_REBOOT, REBOOT_REBOOT, CTYPE_IMMORTAL, CMD_NO_ABBREV, NO_ABIL ),
	SIMPLE_CMD( "recipes", POS_DEAD, do_recipes, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "reclaim", POS_STANDING, do_reclaim, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "recolor", POS_DEAD, do_recolor, 0, CTYPE_UTIL ),
	SCMD_CMD( "redesignate", POS_STANDING, do_designate, NO_MIN, CTYPE_BUILD, SCMD_REDESIGNATE ),
	SCMD_CMD( "reject", POS_DEAD, do_accept, NO_MIN, CTYPE_UTIL, SCMD_REJECT ),
	GRANT_CMD( "reload", POS_DEAD, do_reload, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_RELOAD ),
	SIMPLE_CMD( "remove", POS_RESTING, do_remove, NO_MIN, CTYPE_COMM ),
	STANDARD_CMD( "repair", POS_STANDING, do_maintain, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "reputations", POS_DEAD, do_factions, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "rescale", POS_RESTING, do_rescale, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_RESCALE ),
	SIMPLE_CMD( "respawn", POS_DEAD, do_respawn, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "respond", POS_RESTING, do_respond, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "rest", POS_RESTING, do_rest, NO_MIN, CTYPE_MOVE ),
	GRANT_CMD( "restore", POS_DEAD, do_restore, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_RESTORE ),
	SIMPLE_CMD( "retrieve", POS_STANDING, do_retrieve, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "return", POS_DEAD, do_return, NO_MIN, CTYPE_IMMORTAL ),
	SIMPLE_CMD( "rework", POS_STANDING, do_rework, NO_MIN, CTYPE_SKILL ),
	SIMPLE_CMD( "ride", POS_STANDING, do_mount, NO_MIN, CTYPE_MOVE ),
	SCMD_CMD( "rite", POS_SLEEPING, do_cast, NO_MIN, CTYPE_SKILL, SCMD_RITUAL ),
	SCMD_CMD( "ritual", POS_SLEEPING, do_cast, NO_MIN, CTYPE_SKILL, SCMD_RITUAL ),
	STANDARD_CMD( "roadsign", POS_STANDING, do_roadsign, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "role", POS_DEAD, do_role, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "roll", POS_RESTING, do_roll, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "rollhistory", POS_DEAD, do_history, NO_MIN, CTYPE_COMM, SCMD_ROLL_HISTORY ),
	SIMPLE_CMD( "roster", POS_DEAD, do_roster, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "run", POS_STANDING, do_run, NO_MIN, CTYPE_MOVE ),
	SCMD_CMD( "rent", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_RENT ),
	SCMD_CMD( "report", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_REPORT ),

	SCMD_CMD( "say", POS_RESTING, do_say, NO_MIN, CTYPE_COMM, SCMD_SAY ),
	SCMD_CMD( "'", POS_RESTING, do_say, NO_MIN, CTYPE_COMM, SCMD_SAY ),
	SIMPLE_CMD( "sacrifice", POS_STANDING, do_sacrifice, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "sail", POS_SITTING, do_drive, NO_MIN, NO_GRANTS, SCMD_SAIL, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "save", POS_STUNNED, do_save, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "saw", POS_STANDING, do_saw, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SCMD_CMD( "sayhistory", POS_DEAD, do_history, NO_MIN, CTYPE_COMM, SCMD_SAY_HISTORY ),
	SIMPLE_CMD( "score", POS_DEAD, do_score, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "scan", POS_RESTING, do_scan, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "scrap", POS_STANDING, do_scrap, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ABBREV, NO_ABIL ),
	STANDARD_CMD( "scrape", POS_STANDING, do_scrape, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "search", POS_STANDING, do_search, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "seed", POS_RESTING, do_seed, NO_MIN, CTYPE_EMPIRE ),
	STANDARD_CMD( "selfdelete", POS_SLEEPING, do_selfdelete, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_UTIL, CMD_NO_ABBREV, NO_ABIL ),
	GRANT_CMD( "send", POS_SLEEPING, do_send, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_SEND ),
	SIMPLE_CMD( "separate", POS_RESTING, do_separate, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "set", POS_DEAD, do_set, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_SET ),
	STANDARD_CMD( "sew", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_SEW, CTYPE_BUILD, NOBITS, NO_ABIL ),
	SIMPLE_CMD( "share", POS_RESTING, do_share, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "shear", POS_STANDING, do_shear, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SIMPLE_CMD( "sheathe", POS_RESTING, do_sheathe, NO_MIN, CTYPE_COMBAT ),
	SIMPLE_CMD( "ship", POS_SLEEPING, do_ship, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "shoot", POS_FIGHTING, do_shoot, NO_MIN, CTYPE_COMBAT ),
	SCMD_CMD( "shout", POS_RESTING, do_pub_comm, NO_MIN, CTYPE_COMM, SCMD_SHOUT ),
	SIMPLE_CMD( "show", POS_DEAD, do_show, LVL_START_IMM, CTYPE_IMMORTAL ),
	STANDARD_CMD( "shutdown", POS_DEAD, do_reboot, LVL_CIMPL, GRANT_SHUTDOWN, REBOOT_SHUTDOWN, CTYPE_IMMORTAL, CMD_NO_ABBREV, NO_ABIL ),
	SCMD_CMD( "sip", POS_RESTING, do_drink, NO_MIN, CTYPE_UTIL, SCMD_SIP ),
	SIMPLE_CMD( "sire", POS_STANDING, do_sire, NO_MIN, CTYPE_COMBAT ),
	SIMPLE_CMD( "sit", POS_RESTING, do_sit, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "skills", POS_DEAD, do_skills, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "skin", POS_STANDING, do_skin, NO_MIN, CTYPE_SKILL ),
	SIMPLE_CMD( "sleep", POS_SLEEPING, do_sleep, NO_MIN, CTYPE_MOVE ),
	GRANT_CMD( "slay", POS_RESTING, do_slay, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_SLAY ),
	STANDARD_CMD( "smelt", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_SMELT, CTYPE_BUILD, NOBITS, NO_ABIL ),
	GRANT_CMD( "snoop", POS_DEAD, do_snoop, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_SNOOP ),
	SIMPLE_CMD( "socials", POS_DEAD, do_socials, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "speak", POS_DEAD, do_speak, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "specialize", POS_STANDING, do_specialize, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "split", POS_RESTING, do_split, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "stand", POS_RESTING, do_stand, NO_MIN, CTYPE_MOVE ),
	SCMD_CMD( "stake", POS_FIGHTING, do_stake, NO_MIN, CTYPE_COMBAT, FALSE ),
	SIMPLE_CMD( "start", POS_DEAD, do_start, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "stat", POS_DEAD, do_stat, LVL_START_IMM, CTYPE_IMMORTAL ),
	SCMD_CMD( "statusmessages", POS_DEAD, do_fightmessages, NO_MIN, CTYPE_UTIL, SCMD_STATUS ),
	SCMD_CMD( "smessages", POS_DEAD, do_fightmessages, NO_MIN, CTYPE_UTIL, SCMD_STATUS ),
	SIMPLE_CMD( "steal", POS_STANDING, do_steal, NO_MIN, CTYPE_COMBAT ),
	SIMPLE_CMD( "store", POS_STANDING, do_store, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "stop", POS_DEAD, do_stop, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "struggle", POS_STUNNED, do_struggle, NO_MIN, CTYPE_COMBAT ),
	SIMPLE_CMD( "summary", POS_DEAD, do_summary, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "summon", POS_DEAD, do_summon, NO_MIN, CTYPE_SKILL ),
	SIMPLE_CMD( "survey", POS_STANDING, do_survey, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "swap", POS_RESTING, do_swap, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "swim", POS_STANDING, do_swim, NO_MIN, CTYPE_MOVE ),
	GRANT_CMD( "switch", POS_DEAD, do_switch, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_SWITCH ),
	SIMPLE_CMD( "syslog", POS_DEAD, do_syslog, LVL_START_IMM, CTYPE_IMMORTAL ),

	SIMPLE_CMD( "tell", POS_DEAD, do_tell, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "take", POS_RESTING, do_get, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "tame", POS_STANDING, do_tame, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_SKILL, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "tan", POS_STANDING, do_tan, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_EMPIRE, CMD_NO_ANIMALS, NO_ABIL ),
	SCMD_CMD( "taste", POS_RESTING, do_eat, NO_MIN, CTYPE_UTIL, SCMD_TASTE ),
	GRANT_CMD( "tattach", POS_DEAD, do_tattach, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_TRIGGERS ),
	GRANT_CMD( "tdetach", POS_DEAD, do_tdetach, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_TRIGGERS ),
	GRANT_CMD( "tedit", POS_DEAD, do_tedit, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_TEDIT ),
	SCMD_CMD( "teleport", POS_STANDING, do_goto, LVL_GOD, CTYPE_IMMORTAL, SCMD_TELEPORT ),
	SCMD_CMD( "tellhistory", POS_DEAD, do_history, NO_MIN, CTYPE_COMM, SCMD_TELL_HISTORY ),
	SIMPLE_CMD( "temperature", POS_DEAD, do_temperature, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "territory", POS_DEAD, do_territory, NO_MIN, CTYPE_EMPIRE ),
	STANDARD_CMD( "throw", POS_FIGHTING, do_throw, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMBAT, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "thaw", POS_DEAD, do_wizutil, LVL_CIMPL, GRANT_FREEZE, SCMD_THAW, CTYPE_IMMORTAL, CMD_NO_ANIMALS, NO_ABIL ),
	SCMD_CMD( "tie", POS_STANDING, do_tie, NO_MIN, CTYPE_COMBAT, FALSE ),
	SIMPLE_CMD( "time", POS_DEAD, do_time, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "tips", POS_DEAD, do_tip, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "title", POS_DEAD, do_title, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "toggles", POS_DEAD, do_toggle, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "tomb", POS_DEAD, do_tomb, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "track", POS_STANDING, do_track, NO_MIN, CTYPE_SKILL ),
	SIMPLE_CMD( "trade", POS_RESTING, do_trade, NO_MIN, CTYPE_MOVE ),
	GRANT_CMD( "transfer", POS_SLEEPING, do_trans, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_TRANSFER ),
	SIMPLE_CMD( "transport", POS_STANDING, do_transport, NO_MIN, CTYPE_MOVE ),
	STANDARD_CMD( "tunnel", POS_STANDING, do_tunnel, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_BUILD, CMD_NO_ANIMALS, NO_ABIL ),
	SCMD_CMD( "typo", POS_DEAD, do_gen_write, NO_MIN, CTYPE_COMM, SCMD_TYPO ),

	SCMD_CMD( "unalias", POS_DEAD, do_alias, NO_MIN, CTYPE_UTIL, SCMD_UNALIAS ),
	STANDARD_CMD( "unapprove", POS_DEAD, do_approve, LVL_CIMPL, GRANT_APPROVE, SCMD_UNAPPROVE, CTYPE_IMMORTAL, NOBITS, NO_ABIL ),
	GRANT_CMD( "unbind", POS_SLEEPING, do_unbind, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_UNBIND ),
	STANDARD_CMD( "unfriend", POS_DEAD, do_unfriend, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_UTIL, CMD_NO_ABBREV, NO_ABIL ),
	STANDARD_CMD( "unharness", POS_STANDING, do_unharness, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	SCMD_CMD( "unkeep", POS_DEAD, do_keep, NO_MIN, CTYPE_UTIL, SCMD_UNKEEP ),
	STANDARD_CMD( "unload", POS_STANDING, do_unload_vehicle, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	STANDARD_CMD( "unloadvehicle", POS_STANDING, do_unload_vehicle, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_NO_ANIMALS, NO_ABIL ),
	GRANT_CMD( "unquest", POS_SLEEPING, do_unquest, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_UNQUEST ),
	SCMD_CMD( "unstake", POS_STANDING, do_stake, NO_MIN, CTYPE_COMBAT, TRUE ),
	STANDARD_CMD( "untie", POS_STANDING, do_tie, NO_MIN, NO_GRANTS, TRUE, CTYPE_COMBAT, CMD_NO_ANIMALS, NO_ABIL ),
	GRANT_CMD( "unban", POS_DEAD, do_unban, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_BAN ),
	SIMPLE_CMD( "unpublicize", POS_DEAD, do_unpublicize, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "unpaint", POS_STANDING, do_unpaint, NO_MIN, CTYPE_BUILD ),
	GRANT_CMD( "unprogress", POS_DEAD, do_unprogress, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_UNPROGRESS ),
	SIMPLE_CMD( "unshare", POS_RESTING, do_unshare, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "uptime", POS_DEAD, do_date, LVL_START_IMM, CTYPE_IMMORTAL, SCMD_UPTIME ),
	SIMPLE_CMD( "upgrade", POS_STANDING, do_upgrade, NO_MIN, CTYPE_BUILD ),
	SIMPLE_CMD( "use", POS_RESTING, do_use, NO_MIN, CTYPE_MOVE ),
	GRANT_CMD( "users", POS_DEAD, do_users, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_USERS ),
	SIMPLE_CMD( "utility", POS_DEAD, do_admin_util, LVL_START_IMM, CTYPE_IMMORTAL ),
	SCMD_CMD( "ungroup", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_UNGROUP ),

	SCMD_CMD( "version", POS_DEAD, do_gen_ps, NO_MIN, CTYPE_UTIL, SCMD_VERSION ),
	SIMPLE_CMD( "visible", POS_RESTING, do_visible, NO_MIN, CTYPE_UTIL ),
	GRANT_CMD( "vnum", POS_DEAD, do_vnum, LVL_START_IMM, CTYPE_IMMORTAL, GRANT_OLC ),
	GRANT_CMD( "vstat", POS_DEAD, do_vstat, LVL_START_IMM, CTYPE_IMMORTAL, GRANT_OLC ),
	GRANT_CMD( "vdelete", POS_DEAD, do_vdelete, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_TRIGGERS ),

	SIMPLE_CMD( "wake", POS_SLEEPING, do_wake, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "warehouse", POS_DEAD, do_warehouse, NO_MIN, CTYPE_MOVE ),
	SIMPLE_CMD( "wear", POS_RESTING, do_wear, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "weather", POS_RESTING, do_weather, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "weave", POS_DEAD, do_gen_craft, NO_MIN, NO_GRANTS, CRAFT_TYPE_WEAVE, CTYPE_BUILD, NOBITS, NO_ABIL ),
	STANDARD_CMD( "who", POS_DEAD, do_who, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMM, CMD_STAY_HIDDEN, NO_ABIL ),
	SIMPLE_CMD( "whoami", POS_DEAD, do_whoami, NO_MIN, CTYPE_UTIL ),
	STANDARD_CMD( "whois", POS_DEAD, do_whois, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_COMM, CMD_STAY_HIDDEN, NO_ABIL ),
	SIMPLE_CMD( "where", POS_RESTING, do_where, NO_MIN, CTYPE_COMM ),
	SIMPLE_CMD( "whereami", POS_RESTING, do_whereami, NO_MIN, CTYPE_UTIL ),
	SCMD_CMD( "whisper", POS_RESTING, do_spec_comm, NO_MIN, CTYPE_COMM, SCMD_WHISPER ),
	SIMPLE_CMD( "wield", POS_RESTING, do_wield, NO_MIN, CTYPE_UTIL ),
	SIMPLE_CMD( "withdraw", POS_STANDING, do_withdraw, NO_MIN, CTYPE_EMPIRE ),
	SCMD_CMD( "wiznet", POS_DEAD, do_pub_comm, LVL_START_IMM, CTYPE_IMMORTAL, SCMD_WIZNET ),
	SCMD_CMD( ";", POS_DEAD, do_pub_comm, LVL_START_IMM, CTYPE_IMMORTAL, SCMD_WIZNET ),
	SCMD_CMD( "wizhelp", POS_DEAD, do_commands, LVL_GOD, CTYPE_IMMORTAL, SCMD_WIZHELP ),
	SCMD_CMD( "wizhistory", POS_DEAD, do_history, LVL_GOD, CTYPE_COMM, SCMD_GOD_HISTORY ),
	SCMD_CMD( "wizlist", POS_DEAD, do_gen_text_string, NO_MIN, CTYPE_UTIL, TEXT_FILE_WIZLIST ),
	GRANT_CMD( "wizlock", POS_DEAD, do_wizlock, LVL_CIMPL, CTYPE_IMMORTAL, GRANT_WIZLOCK ),
	SIMPLE_CMD( "workforce", POS_DEAD, do_workforce, NO_MIN, CTYPE_EMPIRE ),
	SIMPLE_CMD( "write", POS_STANDING, do_write, NO_MIN, CTYPE_COMM ),
	SCMD_CMD( "wimpy", POS_DEAD, do_no_cmd, NO_MIN, CTYPE_UTIL, NOCMD_WIMPY ),
	
	{ ",", POS_DEAD, do_string_editor, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_UTIL, NOBITS, NO_ABIL },
	
	/* DG trigger commands */
	STANDARD_CMD( "madventurecomplete", POS_DEAD, do_madventurecomplete, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "maggro", POS_RESTING, do_maggro, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "masound", POS_DEAD, do_masound, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mbuild", POS_DEAD, do_mbuild, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mkill", POS_FIGHTING, do_mkill, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mjunk", POS_DEAD, do_mjunk, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mdamage", POS_DEAD, do_mdamage, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "maoe", POS_DEAD, do_maoe, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mdot", POS_DEAD, do_mdot, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mdoor", POS_DEAD, do_mdoor, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mecho", POS_DEAD, do_mecho, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mechoaround", POS_DEAD, do_mechoaround, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mechoneither", POS_DEAD, do_mechoneither, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "msend", POS_DEAD, do_msend, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mload", POS_DEAD, do_mload, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mlog", POS_DEAD, do_mlog, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mmod", POS_DEAD, do_mmod, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mmorph", POS_DEAD, do_mmorph, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mmove", POS_STANDING, do_mmove, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_MOVE, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mpurge", POS_DEAD, do_mpurge, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mquest", POS_DEAD, do_mquest, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mgoto", POS_DEAD, do_mgoto, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mat", POS_DEAD, do_mat, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mown", POS_DEAD, do_mown, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mrestore", POS_DEAD, do_mrestore, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mscale", POS_DEAD, do_mscale, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "msiege", POS_DEAD, do_msiege, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mslay", POS_DEAD, do_mslay, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mteleport", POS_DEAD, do_mteleport, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mterracrop", POS_DEAD, do_mterracrop, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mterraform", POS_DEAD, do_mterraform, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mforce", POS_DEAD, do_mforce, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mheal", POS_DEAD, do_mheal, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mhunt", POS_DEAD, do_mhunt, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mremember", POS_DEAD, do_mremember, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mforget", POS_DEAD, do_mforget, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mtransform", POS_DEAD, do_mtransform, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mbuildingecho", POS_DEAD, do_mbuildingecho, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mvehicleecho", POS_DEAD, do_mvehicleecho, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mregionecho", POS_DEAD, do_mregionecho, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "msubecho", POS_DEAD, do_msubecho, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),
	STANDARD_CMD( "mfollow", POS_DEAD, do_mfollow, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_IMMORTAL, CMD_IMM_OR_MOB_ONLY | CMD_STAY_HIDDEN, NO_ABIL ),

	/* this must be last */
	STANDARD_CMD( "\n", POS_DEAD, NULL, NO_MIN, NO_GRANTS, NO_SCMD, CTYPE_UTIL, NOBITS, NO_ABIL )
};


 //////////////////////////////////////////////////////////////////////////////
//// COMMAND INTERPRETER /////////////////////////////////////////////////////

/*
 * This is the actual command interpreter called from game_loop() in comm.c
 * It makes sure you are the proper level and position to execute the command,
 * then calls the appropriate function.
 */
void command_interpreter(char_data *ch, char *argument) {
	int cmd, length, iter;
	char arg[MAX_INPUT_LENGTH], *line;

	/* just drop to next line for hitting CR */
	skip_spaces(&argument);
	if (!*argument)
		return;

#ifdef BIG_BROTHER
	if (!IS_NPC(ch)) {
		printf("prool debug: '%s' '%s'\n", GET_NAME(ch), argument);
		fflush(0);
	}
#endif

	/*
	 * special case to handle one-character, non-alphanumeric commands;
	 * requested by many people so "'hi" or ";godnet test" is possible.
	 * Patch sent by Eric Green and Stefan Wasilewski.
	 *
	 * Now ignores '!' which cannot be typed by players (it causes a repeated
	 * command at a lower code level). This allows ! to be used for npc-only
	 * command triggers. -paul 12/6/2025
	 */
	if (!isalpha(*argument) && *argument != '!') {
		arg[0] = argument[0];
		arg[1] = '\0';
		line = argument + 1;
	}
	else {
		line = any_one_arg(argument, arg);
	}
	
	// lowercase arg to speed up command comparisons
	for (iter = 0; iter < strlen(arg); ++iter) {
		arg[iter] = LOWER(arg[iter]);
	}

	// Command trigger (1/3): exact match on typed-in word (note: this is checked 2 more times below, AND inside abilities and socials)s
	if (check_command_trigger(ch, arg, line, CMDTRG_EXACT)) {
		return;
	}

	/* otherwise, find the command */
	for (length = strlen(arg), cmd = 0; *cmd_info[cmd].command != '\n'; ++cmd) {
		if (GET_ACCESS_LEVEL(ch) < cmd_info[cmd].minimum_level && (cmd_info[cmd].grants == NO_GRANTS || !IS_GRANTED(ch, cmd_info[cmd].grants))) {
			continue;
		}
		if (IS_SET(cmd_info[cmd].flags, CMD_NO_ABBREV) ? strcmp(arg, cmd_info[cmd].command) : strncmp(cmd_info[cmd].command, arg, length)) {
			continue;
		}		
		if (IS_SET(cmd_info[cmd].flags, CMD_VAMPIRE_ONLY) && !IS_VAMPIRE(ch)) {
			continue;
		}
		if (IS_SET(cmd_info[cmd].flags, CMD_IMM_OR_MOB_ONLY) && GET_ACCESS_LEVEL(ch) < LVL_START_IMM && !IS_NPC(ch)) {
			continue;
		}
		// NPCs can use ability commands IF they aren't charmed; players require the ability
		if (cmd_info[cmd].ability != NO_ABIL && (IS_NPC(ch) ? AFF_FLAGGED(ch, AFF_CHARM) : !has_ability(ch, cmd_info[cmd].ability))) {
			continue;
		}
		
		// found!
		break;
	}
	
	// check abilities and socials
	if (*cmd_info[cmd].command == '\n' && check_ability(ch, argument, FALSE)) {
		return;
	}
	else if (*cmd_info[cmd].command == '\n' && check_social(ch, argument, FALSE)) {
		return;
	}
	else if (strlen(arg) < strlen(cmd_info[cmd].command) && check_ability(ch, argument, TRUE)) {
		// If the player abbreviated the actual command, and what they typed is an exact match for a ability, do the ability instead.
		return;
	}
	else if (strlen(arg) < strlen(cmd_info[cmd].command) && check_social(ch, argument, TRUE)) {
		// If the player abbreviated the actual command ("nod" for
		// nodismantle), and what they typed is an exact match for a social,
		// do the social instead.
		return;
	}
	
	// we are locked in, not doing a social, not doing an ability: now reveal hidden
	if (AFF_FLAGGED(ch, AFF_HIDDEN) && !IS_SET(cmd_info[cmd].flags, CMD_STAY_HIDDEN | CMD_UNHIDE_AFTER)) {
		REMOVE_BIT(AFF_FLAGS(ch), AFF_HIDDEN);
		affects_from_char_by_aff_flag(ch, AFF_HIDDEN, FALSE);
	}
	
	if (*cmd_info[cmd].command == '\n') {
		// Command trigger (2/3): abbrev match on non-matching command
		if (check_command_trigger(ch, arg, line, CMDTRG_ABBREV)) {
			return;
		}
		// otherwise, no match

#include "prool-patch.c"

		send_config_msg(ch, "huh_string");
	}
	else if (!char_can_act(ch, cmd_info[cmd].minimum_position, !IS_SET(cmd_info[cmd].flags, CMD_NO_ANIMALS), (cmd_info[cmd].ctype != CTYPE_COMBAT && cmd_info[cmd].ctype != CTYPE_SKILL && cmd_info[cmd].ctype != CTYPE_BUILD), IS_SET(cmd_info[cmd].flags, CMD_WHILE_FEEDING))) {
		// sent own error message
	}
	else if (GET_FEEDING_FROM(ch) && cmd_info[cmd].minimum_position >= POS_SLEEPING && !IS_SET(cmd_info[cmd].flags, CMD_WHILE_FEEDING)) {
		msg_to_char(ch, "You can't do that while feeding!\r\n");
	}
	else if (IS_NPC(ch) && cmd_info[cmd].minimum_level >= LVL_GOD) {
		send_to_char("You can't use immortal commands while switched.\r\n", ch);
	}
	else if (IS_SET(cmd_info[cmd].flags, CMD_NOT_RP) && !IS_NPC(ch) && !IS_GOD(ch) && !IS_IMMORTAL(ch) && PRF_FLAGGED(ch, PRF_RP)) {
		msg_to_char(ch, "You can't do that while role-playing!\r\n");
	}
	else if (cmd_info[cmd].command_pointer == NULL) {
		send_to_char("Sorry, that command hasn't been implemented yet.\r\n", ch);
	}
	
	// Command trigger (3/3): exact match on abbreviated command
	else if (strlen(arg) < strlen(cmd_info[cmd].command) && check_command_trigger(ch, (char*)cmd_info[cmd].command, line, CMDTRG_EXACT)) {
		return;
	}
	
	else {
		((*cmd_info[cmd].command_pointer) (ch, line, cmd, cmd_info[cmd].subcmd));
	}

	/* Unhide after ? */
	if (ch && IS_SET(cmd_info[cmd].flags, CMD_UNHIDE_AFTER)) {
		REMOVE_BIT(AFF_FLAGS(ch), AFF_HIDDEN);
		affects_from_char_by_aff_flag(ch, AFF_HIDDEN, FALSE);
	}
}


/* Used in specprocs, mostly.  (Exactly) matches "command" to cmd number */
int find_command(const char *command) {
	int cmd;

	for (cmd = 0; *cmd_info[cmd].command != '\n'; cmd++)
	if (!str_cmp(cmd_info[cmd].command, command))
		return (cmd);

	return (-1);
}


 //////////////////////////////////////////////////////////////////////////////
//// ALIAS SYSTEM ////////////////////////////////////////////////////////////

struct alias_data *find_alias(struct alias_data *alias_list, char *str) {
	while (alias_list != NULL) {
		if (LOWER(*str) == LOWER(*alias_list->alias))	/* hey, every little bit counts :-) */
			if (!str_cmp(str, alias_list->alias))
				return (alias_list);

		alias_list = alias_list->next;
	}

	return (NULL);
}


void free_alias(struct alias_data *a) {
	if (a->alias)
		free(a->alias);
	if (a->replacement)
		free(a->replacement);
	free(a);
}


/*
 * Valid numeric replacements are only $1 .. $9 (makes parsing a little
 * easier, and it's not that much of a limitation anyway.)  Also valid
 * is "$*", which stands for the entire original line after the alias.
 * ";" is used to delimit commands.
 */
#define NUM_TOKENS       9

void perform_complex_alias(struct txt_q *input_q, char *orig, struct alias_data *a) {
	char buf[MAX_STRING_LENGTH];
	struct txt_q temp_queue;
	char *tokens[NUM_TOKENS], *temp, *write_point;
	int num_of_tokens = 0, num;

	skip_spaces(&orig);

	/* First, parse the original string */
	temp = strtok(strcpy(buf2, orig), " ");
	while (temp != NULL && num_of_tokens < NUM_TOKENS) {
		tokens[num_of_tokens++] = temp;
		temp = strtok(NULL, " ");
	}

	/* initialize */
	write_point = buf;
	temp_queue.head = temp_queue.tail = NULL;

	/* now parse the alias */
	for (temp = a->replacement; *temp; ++temp) {
		if (*temp == ALIAS_SEP_CHAR) {
			*write_point = '\0';
			buf[MAX_INPUT_LENGTH - 1] = '\0';
			write_to_q(buf, &temp_queue, 1, FALSE);
			write_point = buf;
		}
		else if (*temp == ALIAS_VAR_CHAR) {
			// advance it by 1
			if (!*(++temp)) {
				// hit the end
				break;
			}
			else if ((num = *temp - '1') < num_of_tokens && num >= 0) {
				strcpy(write_point, tokens[num]);
				write_point += strlen(tokens[num]);
			}
			else if (num >= 0 && num >= num_of_tokens) {
				// no arg, just skip it
			}
			else if (*temp == ALIAS_GLOB_CHAR) {
				strcpy(write_point, orig);
				write_point += strlen(orig);
			}
			else if ((*(write_point++) = *temp) == '$') {	/* redouble $ for act safety */
				*(write_point++) = '$';
			}
		}
		else {
			*(write_point++) = *temp;
		}
	}

	*write_point = '\0';
	buf[MAX_INPUT_LENGTH - 1] = '\0';
	write_to_q(buf, &temp_queue, 1, FALSE);

	/* push our temp_queue on to the _front_ of the input queue */
	if (input_q->head == NULL)
		*input_q = temp_queue;
	else {
		temp_queue.tail->next = input_q->head;
		input_q->head = temp_queue.head;
	}
}


/*
 * Given a character and a string, perform alias replacement on it.
 *
 * Return values:
 *   0: String was modified in place; call command_interpreter immediately.
 *   1: String was _not_ modified in place; rather, the expanded aliases
 *      have been placed at the front of the character's input queue.
 */
int perform_alias(descriptor_data *d, char *orig) {
	char first_arg[MAX_INPUT_LENGTH], *ptr;
	struct alias_data *a, *tmp;

	/* Mobs don't have aliases. */
	if (IS_NPC(d->character))
		return (0);

	/* bail out immediately if the guy doesn't have any aliases */
	if ((tmp = GET_ALIASES(d->character)) == NULL)
		return (0);

	/* find the alias we're supposed to match */
	ptr = any_one_arg(orig, first_arg);

	/* bail out if it's null */
	if (!*first_arg)
		return (0);

	/* if the first arg is not an alias, return without doing anything */
	if ((a = find_alias(tmp, first_arg)) == NULL)
		return (0);

	if (a->type == ALIAS_SIMPLE) {
		strcpy(orig, a->replacement);
		return (0);
	}
	else {
		perform_complex_alias(&d->input, ptr, a);
		return (1);
	}
}


/* The interface to the outside world: do_alias / do_unalias */
ACMD(do_alias) {
	char *repl;
	struct alias_data *a;

	if (IS_NPC(ch))
		return;

	repl = any_one_arg(argument, arg);

	if (!*arg) {			/* no argument specified -- list currently defined aliases */
		build_page_display(ch, "Currently defined aliases:");
		if (GET_ALIASES(ch) == NULL) {
			build_page_display(ch, " None.");
		}
		else {
			LL_FOREACH(GET_ALIASES(ch), a) {
				build_page_display(ch, "%-15s %s", a->alias, show_color_codes(a->replacement));
			}
		}
		send_page_display(ch);
	}
	else {			/* otherwise, add or remove aliases */
		/* is this an alias we've already defined? */
		a = find_alias(GET_ALIASES(ch), arg);
		
		// only delete if it's being replaced or they used unalias
		if (a != NULL && (*repl || subcmd == SCMD_UNALIAS)) {
			LL_DELETE(GET_ALIASES(ch), a);
			free_alias(a);
		}
		/* if no replacement string is specified (or they used unalias): */
		if (!*repl || subcmd == SCMD_UNALIAS) {
			if (a == NULL) {
				send_to_char("No such alias.\r\n", ch);
			}
			else if (subcmd == SCMD_UNALIAS) {
				send_to_char("Alias deleted.\r\n", ch);
			}
			else {	// just viewing the alias
				msg_to_char(ch, "%s: %s\r\n", a->alias, a->replacement);
			}
		}
		else {			/* otherwise, either add or redefine an alias */
			if (!str_cmp(arg, "alias")) {
				send_to_char("You can't alias 'alias'.\r\n", ch);
				return;
			}
			CREATE(a, struct alias_data, 1);
			a->alias = str_dup(arg);
			delete_doubledollar(repl);
			a->replacement = str_dup(repl);
			if (strchr(repl, ALIAS_SEP_CHAR) || strchr(repl, ALIAS_VAR_CHAR)) {
				a->type = ALIAS_COMPLEX;
			}
			else {
				a->type = ALIAS_SIMPLE;
			}
			LL_PREPEND(GET_ALIASES(ch), a);
			send_to_char("Alias added.\r\n", ch);
		}
	}
}


 //////////////////////////////////////////////////////////////////////////////
//// HELPER FUNCTIONS ////////////////////////////////////////////////////////

/**
* Checks that a player can take a certain action (like a command).
*
* @param char_data *ch The character trying to act.
* @param int min_pos The minimum allowed POS_ const.
* @param bool allow_animal If FALSE, players can't do this in an animal morph.
* @param bool allow_invulnerable If FALSE, players can't do this while un-attackable.
* @param bool override_feeding If TRUE, ignores position for feeding vampires.
* @return bool TRUE if the character can act, FALSE (with error msg) if not
*/
bool char_can_act(char_data *ch, int min_pos, bool allow_animal, bool allow_invulnerable, bool override_feeding) {
	if (!IS_NPC(ch) && ACCOUNT_FLAGGED(ch, ACCT_FROZEN)) {
		send_to_char("You try, but the mind-numbing cold prevents you...\r\n", ch);
	}
	else if (!allow_animal && CHAR_MORPH_FLAGGED(ch, MORPHF_ANIMAL)) {
		msg_to_char(ch, "You can't do that in this form!\r\n");
	}
	else if (IS_INJURED(ch, INJ_STAKED) && min_pos >= POS_SLEEPING && !IS_IMMORTAL(ch)) {
		msg_to_char(ch, "You can't do that while staked!\r\n");
	}
	else if (AFF_FLAGGED(ch, AFF_STUNNED | AFF_HARD_STUNNED) && min_pos >= POS_SLEEPING && !IS_IMMORTAL(ch)) {
		msg_to_char(ch, "You can't do that while stunned!\r\n");
	}
	else if (AFF_FLAGGED(ch, AFF_EARTHMELDED) && min_pos >= POS_SLEEPING) {
		msg_to_char(ch, "You can't do that while earthmelded.\r\n");
	}
	else if (AFF_FLAGGED(ch, AFF_MUMMIFIED) && min_pos >= POS_SLEEPING) {
		msg_to_char(ch, "You can't do that while mummified.\r\n");
	}
	else if (AFF_FLAGGED(ch, AFF_DEATHSHROUDED) && min_pos >= POS_SLEEPING) {
		msg_to_char(ch, "You can't do that while deathshrouded!\r\n");
	}
	else if (GET_FEEDING_FROM(ch) && min_pos >= POS_SLEEPING && !override_feeding) {
		msg_to_char(ch, "You can't do that right now!\r\n");
	}
	else if (GET_FED_ON_BY(ch) && min_pos >= POS_SLEEPING) {
		msg_to_char(ch, "The ecstasy of the fangs in your flesh is too enchanting to do that...\r\n");
	}
	else if (IS_INJURED(ch, INJ_TIED) && min_pos >= POS_SLEEPING) {
		msg_to_char(ch, "You're tied up!\r\n");
	}
	else if (!allow_invulnerable && AFF_FLAGGED(ch, AFF_NO_ATTACK) && !IS_NPC(ch)) {
		msg_to_char(ch, "You can't do that in this state.\r\n");
	}
	else if (GET_POS(ch) < min_pos) {
		send_low_pos_msg(ch);
	}
	else {
		return TRUE;	// success
	}
	
	// must have failed
	return FALSE;
}


/**
* Sends a message telling the character their position is too low to perform
* some task.
*
* Usage:
*   if (GET_POS(ch) < required_pos) {
*     send_low_pos_msg(ch);
*     return;
*   }
*
* @param char_data *ch The player receiving the error.
*/
void send_low_pos_msg(char_data *ch) {
	switch (GET_POS(ch)) {
		case POS_DEAD: {
			msg_to_char(ch, "Lie still; you are DEAD!!!\r\n");
			msg_to_char(ch, "(Type 'respawn' to come back at your tomb.)\r\n");
			break;
		}
		case POS_INCAP:
		case POS_MORTALLYW: {
			send_to_char("You are in a pretty bad shape, unable to do anything!\r\n", ch);
			break;
		}
		case POS_STUNNED: {
			send_to_char("All you can do right now is think about the stars!\r\n", ch);
			break;
		}
		case POS_SLEEPING: {
			send_to_char("In your dreams, or what?\r\n", ch);
			break;
		}
		case POS_RESTING: {
			send_to_char("Nah... You feel too relaxed to do that.\r\n", ch);
			break;
		}
		case POS_SITTING: {
			send_to_char("Maybe you should get on your feet first?\r\n", ch);
			break;
		}
		case POS_FIGHTING: {
			send_to_char("No way! You're fighting for your life!\r\n", ch);
			break;
		}
		// nothing to send at higher positions
	}
}


 //////////////////////////////////////////////////////////////////////////////
//// COMMAND FUNCTIONS ///////////////////////////////////////////////////////

struct sort_struct {
	int sort_pos;
	byte is_social;
} *cmd_sort_info = NULL;
int num_of_cmds;

void sort_commands(void) {
	int a, b, tmp;

	num_of_cmds = 0;

	/*
	 * first, count commands (num_of_commands is actually one greater than the
	 * number of commands; it inclues the '\n'.
	 */
	while (*cmd_info[num_of_cmds].command != '\n')
		num_of_cmds++;

	/* create data array */
	CREATE(cmd_sort_info, struct sort_struct, num_of_cmds);

	/* initialize it */
	for (a = 1; a < num_of_cmds; a++) {
		cmd_sort_info[a].sort_pos = a;
		cmd_sort_info[a].is_social = 0;
	}

	/* Sort.  'a' starts at 1, not 0, to remove 'RESERVED' */
	for (a = 1; a < num_of_cmds - 1; a++) {
		for (b = a + 1; b < num_of_cmds; b++) {
			if (str_cmp(cmd_info[cmd_sort_info[a].sort_pos].command, cmd_info[cmd_sort_info[b].sort_pos].command) > 0) {
				tmp = cmd_sort_info[a].sort_pos;
				cmd_sort_info[a].sort_pos = cmd_sort_info[b].sort_pos;
				cmd_sort_info[b].sort_pos = tmp;
			}
		}
	}
}


/**
* Frees the command sort data during a "shutdown complete".
*/
void free_command_sort(void) {
	free(cmd_sort_info);
	cmd_sort_info = NULL;
	num_of_cmds = 0;
}


ACMD(do_commands) {
	int i, cmd_num;
	int wizhelp = 0;
	char_data *vict;

	one_argument(argument, arg);

	if (*arg) {
		if (!(vict = get_char_vis(ch, arg, NULL, FIND_CHAR_WORLD))) {
			send_to_char("Who is that?\r\n", ch);
			return;
		}
		if (IS_NPC(vict)) {
			send_to_char("You can't check the commands of an NPC.\r\n", ch);
			return;
		}
		if (GET_ACCESS_LEVEL(ch) < GET_ACCESS_LEVEL(vict)) {
			send_to_char("You can't see the commands of people above your access level.\r\n", ch);
			return;
		}
	}
	else
		vict = ch;

	if (subcmd == SCMD_WIZHELP)
		wizhelp = 1;

	build_page_display(ch, "The following %s%s are available to %s:", wizhelp ? "privileged " : "", "commands", vict == ch ? "you" : PERS(vict, ch, 1));

	/* cmd_num starts at 1, not 0, to remove 'RESERVED' */
	for (cmd_num = 1; cmd_num < num_of_cmds; cmd_num++) {
		i = cmd_sort_info[cmd_num].sort_pos;
		if (cmd_info[i].minimum_level >= 0 && (cmd_info[i].ability == NO_ABIL || has_ability(vict, cmd_info[i].ability)) && (GET_ACCESS_LEVEL(vict) >= cmd_info[i].minimum_level || (cmd_info[i].grants != NO_GRANTS && IS_GRANTED(vict, cmd_info[i].grants))) && (cmd_info[i].minimum_level >= LVL_GOD) == wizhelp) {
			if (!IS_SET(cmd_info[i].flags, CMD_IMM_OR_MOB_ONLY) || GET_ACCESS_LEVEL(vict) >= LVL_START_IMM || IS_NPC(vict)) {
				build_page_display_col_str(ch, 5, FALSE, cmd_info[i].command);
			}
		}
	}
	
	send_page_display(ch);
}


ACMD(do_missing_help_files) {
	struct help_index_element *found;
	ability_data *abil, *next_abil;
	skill_data *skill, *next_skill;
	int iter, count;
	
	count = 0;
	
	// commands:
	for (iter = 0; *cmd_info[iter].command != '\n'; ++iter) {
		if (strcmp(cmd_info[iter].command, "RESERVED") != 0) {
			found = find_help_entry(LVL_TOP, cmd_info[iter].command);
		
			if (!found) {
				++count;
				build_page_display_col_str(ch, 4, FALSE, cmd_info[iter].command);
			}
		}
	}
	
	// skills
	HASH_ITER(hh, skill_table, skill, next_skill) {
		if (IS_SET(SKILL_FLAGS(skill), SKILLF_IN_DEVELOPMENT)) {
			continue;	// don't count if in-dev
		}
		if (!SKILL_NAME(skill)) {
			continue;
		}
		
		if (!find_help_entry(LVL_TOP, SKILL_NAME(skill))) {
			++count;
			build_page_display_col_str(ch, 4, FALSE, SKILL_NAME(skill));
		}
		if (SKILL_ABBREV(skill) && *SKILL_ABBREV(skill) && !find_help_entry(LVL_TOP, SKILL_ABBREV(skill))) {
			++count;
			build_page_display_col_str(ch, 4, FALSE, SKILL_ABBREV(skill));
		}
	}
	
	// abilities
	HASH_ITER(hh, ability_table, abil, next_abil) {
		if (!ABIL_NAME(abil)) {
			continue;
		}
		if (has_ability_data_any(abil, ADL_PARENT)) {
			continue;	// generally not needed
		}
		
		if (!find_help_entry(LVL_TOP, ABIL_NAME(abil))) {
			++count;
			build_page_display_col_str(ch, 4, FALSE, ABIL_NAME(abil));
		}
		if (ABIL_COMMAND(abil) && *ABIL_COMMAND(abil) && str_cmp(ABIL_COMMAND(abil), ABIL_NAME(abil)) && !find_help_entry(LVL_TOP, ABIL_COMMAND(abil))) {
			++count;
			build_page_display_col_str(ch, 4, FALSE, ABIL_COMMAND(abil));
		}
	}
	
	if (count == 0) {
		msg_to_char(ch, "Everything appears to have help files (but some may just be abbreviations).\r\n");
		clear_page_display(ch);	// should be empty, but this is the Justin Case
	}
	else {
		build_page_display_prepend(ch, "The following things need help files:");
		send_page_display(ch);
	}
}


 //////////////////////////////////////////////////////////////////////////////
//// CHARACTER CREATION //////////////////////////////////////////////////////

// for character creation: order of creation
struct {
	int state;
} creation_data[] = {
	{ CON_NEWPASSWD },
	{ CON_CNFPASSWD },
	{ CON_QLAST_NAME },	// skips to CON_QSEX sometimes
	{ CON_SLAST_NAME },
	{ CON_CLAST_NAME },
	{ CON_QSEX },
	{ CON_Q_SCREEN_READER },	// skips to CON_Q_HAS_ALT
	
	{ CON_Q_HAS_ALT },	// skips to CON_Q_ARCHETYPE
	{ CON_Q_ALT_NAME },
	{ CON_Q_ALT_PASSWORD },
	
	{ CON_Q_ARCHETYPE },	// skips to CON_BONUS_TRAIT if no archetypes exist
	{ CON_ARCHETYPE_CNFRM },
	
	{ CON_BONUS_TRAIT },	// skips if there's no 0-hour first bonus trait
	
	{ CON_PROMO_CODE },
	{ CON_CONFIRM_PROMO_CODE },	// only if given invalid code
	
	{ CON_REFERRAL },	// skipped if alt
	{ CON_FINISH_CREATION },
	
	// put this last
	{ NOTHING }
};


/**
* Sends a character creation prompt based on current state
*
* @param descriptor_data *d the user
*/
void prompt_creation(descriptor_data *d) {
	switch (STATE(d)) {
		case CON_Q_SCREEN_READER: {
			msg_to_desc(d, "\r\n%s makes heavy use of an ascii map, but also supports screen\r\n", config_get_string("mud_name"));
			SEND_TO_Q("readers for the visually impaired. This will replace the map with a short\r\n", d);
			SEND_TO_Q("description of what you can see in each direction on the world map. This\r\n", d);
			SEND_TO_Q("option is only recommended for players using screen readers. You can see\r\n", d);
			SEND_TO_Q("HELP SCREEN READER once you're in the game for more information.\r\n", d);
			msg_to_desc(d, "\r\nAre you using a screen reader (y/n)? %s", telnet_go_ahead(d));
			break;
		}
		case CON_Q_HAS_ALT: {
			SEND_TO_Q("\r\n&cMultiple characters and Alts:&0\r\n", d);
			SEND_TO_Q("If you have an existing character on this game, you must link this new\r\n", d);
			SEND_TO_Q("character to your other one. This is not optional. We require that you link all\r\n", d);
			SEND_TO_Q("your characters together so that the game can process you as the same person.\r\n", d);
			SEND_TO_Q("Other players will NOT be informed who your alts are. Only immortals will know.\r\n", d);
			SEND_TO_Q("\r\n", d);
			msg_to_desc(d, "Do you have an existing character (y/n)? %s", telnet_go_ahead(d));
			break;
		}
		case CON_Q_ALT_NAME: {
			msg_to_desc(d, "\r\nEnter the name of any one of your other characters (leave blank to cancel): %s", telnet_go_ahead(d));
			break;
		}
		case CON_Q_ALT_PASSWORD: {
			msg_to_desc(d, "\r\nEnter the password for that character: %s", telnet_go_ahead(d));
			ProtocolNoEcho(d, true);
			break;
		}
		case CON_NEWPASSWD: {
			SEND_TO_Q("New character.\r\n\r\n", d);
			msg_to_desc(d, "Give me a password for %s: %s", GET_PC_NAME(d->character), telnet_go_ahead(d));
			ProtocolNoEcho(d, true);
			break;
		}
		case CON_CNFPASSWD: {
			msg_to_desc(d, "\r\n\r\nPlease retype password: %s", telnet_go_ahead(d));
			break;
		}
		case CON_QLAST_NAME: {
			if (!IS_SET(config_get_bitvector("lastname_mode"), LASTNAME_SET_AT_CREATION)) {
				// not allowed to set a lastname at creation
				set_creation_state(d, CON_QSEX);
				break;
			}
			
			msg_to_desc(d, "\r\n\r\nWould you like a last name (y/n)? %s", telnet_go_ahead(d));
			break;
		}
		case CON_SLAST_NAME: {
			const char *rules = config_get_string("name_rules_lastname");
			if (rules && *rules) {
				msg_to_desc(d, "\r\n%s", rules);
			}
			msg_to_desc(d, "\r\nEnter your last name: %s", telnet_go_ahead(d));
			break;
		}
		case CON_CLAST_NAME: {
			msg_to_desc(d, "\r\nDid I get that name right, %s %s%s (y/n)? %s", GET_PC_NAME(d->character), GET_PERSONAL_LASTNAME(d->character), (UPPER(*GET_PERSONAL_LASTNAME(d->character)) != *GET_PERSONAL_LASTNAME(d->character)) ? " (first letter is not capitalized)" : "", telnet_go_ahead(d));
			break;
		}
		case CON_QSEX: {
			msg_to_desc(d, "\r\nWhat is your sex (M/F)? %s", telnet_go_ahead(d));
			break;
		}
		case CON_Q_ARCHETYPE: {
			if (archetype_menu[0].type != NOTHING) {
				SUBMENU(d) = 0;
				parse_archetype_menu(d, "");
			}
			else {
				// no archetypes for some reason?
				set_creation_state(d, CON_BONUS_TRAIT);
			}
			break;
		}
		case CON_ARCHETYPE_CNFRM: {
			int iter, sub;
			
			msg_to_desc(d, "\r\nYou chose the following archetypes:\r\n");
			for (iter = 0; iter < NUM_ARCHETYPE_TYPES; ++iter) {
				archetype_data *arch = archetype_proto(CREATION_ARCHETYPE(d->character, iter));
				if (arch) {
					// find menu posm for name
					for (sub = 0; archetype_menu[sub].type != NOTHING; ++sub) {
						if (archetype_menu[sub].type == iter) {
							break;
						}
					}
					
					msg_to_desc(d, "%s: \tc%s\t0 - %s\r\n", archetype_menu[sub].name, GET_ARCH_NAME(arch), GET_ARCH_DESC(arch));
				}
			}
			
			msg_to_desc(d, "\r\nIs this correct (y/n)? %s", telnet_go_ahead(d));
			break;
		}
		case CON_PROMO_CODE: {
			msg_to_desc(d, "\r\nIf you have a promo code, enter it now. Otherwise, just leave it blank > %s", telnet_go_ahead(d));
			break;
		}
		case CON_CONFIRM_PROMO_CODE: {
			msg_to_desc(d, "\r\nUnknown promo code. Proceed without one (y/n)? %s", telnet_go_ahead(d));
			break;
		}
		case CON_REFERRAL: {
			if (!GET_REFERRED_BY(d->character)) {
				msg_to_desc(d, "\r\nWhere did you hear about us (optional, but please mention which website or friend): %s", telnet_go_ahead(d));
			}
			else {
				next_creation_step(d);
			}
			break;
		}
		case CON_FINISH_CREATION: {
			msg_to_desc(d, "\r\n*** Press ENTER: %s", telnet_go_ahead(d));
			break;
		}
		case CON_BONUS_TRAIT: {
			if (num_earned_bonus_traits(d->character) > count_bits(GET_BONUS_TRAITS(d->character))) {
				show_bonus_trait_menu(d->character);
			}
			else if (GET_ACCESS_LEVEL(d->character) > 0) {
				// existing: send player to game
				send_login_motd(d, GET_BAD_PWS(d->character));
				msg_to_desc(d, "\r\n*** Press ENTER: %s", telnet_go_ahead(d));
				STATE(d) = CON_RMOTD;
			}
			else {
				// creating
				next_creation_step(d);
			}
			break;
		}
	}
}


/**
* Advances the user to the next creation step, and prompts
*
* @param descriptor_data *d the user
*/
void next_creation_step(descriptor_data *d) {
	int iter, found = NOTHING;
	
	for (iter = 0; found == NOTHING && creation_data[iter].state != NOTHING; ++iter) {
		if (STATE(d) == creation_data[iter].state) {
			// found the current state, pick the next one
			found = iter+1;
		}
	}
	
	if (found == NOTHING || creation_data[found].state == NOTHING) {
		set_creation_state(d, CON_FINISH_CREATION);
	}
	else {
		set_creation_state(d, creation_data[found].state);
	}
}


// Handler for CON_Q_ALT_NAME
void process_alt_name(descriptor_data *d, char *arg) {
	player_index_data *index;
	
	if (!*arg) {
		// nothing -- send back to has-alt
		set_creation_state(d, CON_Q_HAS_ALT);
	}
	else if ((index = find_player_index_by_name(arg))) {
		GET_CREATION_ALT_ID(d->character) = index->idnum;
		next_creation_step(d);
	}
	else {
		msg_to_desc(d, "Unable to load character '%s'...\r\nPlease enter a valid alt name or leave blank to cancel: %s", arg, telnet_go_ahead(d));
	}
}


// Handler for CON_Q_ALT_PASSWORD
void process_alt_password(descriptor_data *d, char *arg) {	
	char_data *alt = NULL;
	bool file = FALSE, save = FALSE;
	player_index_data *index;

	if (!*arg) {
		// nothing -- send back to has-alt
		set_creation_state(d, CON_Q_HAS_ALT);
	}	
	else if ((index = find_player_index_by_idnum(GET_CREATION_ALT_ID(d->character))) && (alt = find_or_load_player(index->name, &file))) {
		// loaded char, now check password
		if (strncmp(CRYPT(arg, PASSWORD_SALT), GET_PASSWD(alt), MAX_PWD_LENGTH)) {
			syslog(SYS_LOGIN, 0, TRUE, "BAD PW: unable to register alt %s for %s [%s]", GET_NAME(d->character), GET_NAME(alt), d->host);
			GET_BAD_PWS(alt)++;
			save = TRUE;

			if (++(d->bad_pws) >= config_get_int("max_bad_pws")) {	/* 3 strikes and you're out. */
				SEND_TO_Q("Wrong password... disconnecting.\r\n", d);
				STATE(d) = CON_CLOSE;
			}
			else {
				msg_to_desc(d, "Wrong password.\r\nPassword: %s", telnet_go_ahead(d));
			}
		}
		else {	// password ok
			ProtocolNoEcho(d, false);
			syslog(SYS_LOGIN, 0, TRUE, "NEW: associating new user %s with account for %s", GET_NAME(d->character), GET_NAME(alt));
			
			// does 2nd player have an account already? if not, make one
			if (!GET_ACCOUNT(alt)) {
				create_account_for_player(alt);
				save = TRUE;
			}
			GET_TEMPORARY_ACCOUNT_ID(d->character) = GET_ACCOUNT(alt)->id;
			
			// copy referral
			if ((!GET_REFERRED_BY(d->character) || !*GET_REFERRED_BY(d->character)) && GET_REFERRED_BY(alt)) {
				if (GET_REFERRED_BY(d->character)) {
					free(GET_REFERRED_BY(d->character));
				}
				GET_REFERRED_BY(d->character) = str_dup(GET_REFERRED_BY(alt));
			}
			
			// copy no-friends pref
			if (PRF_FLAGGED(alt, PRF_NO_FRIENDS)) {
				SET_BIT(PRF_FLAGS(d->character), PRF_NO_FRIENDS);
			}
			
			next_creation_step(d);
		}
		
		// need to take char of "alt"
		if (save) {
			if (file) {
				store_loaded_char(alt);
				file = FALSE;
				alt = NULL;
			}
			else {
				SAVE_CHAR(alt);
			}
		}
		
		if (file && alt) {
			free_char(alt);
		}
		
		// state was set above
	}
	else {
		msg_to_desc(d, "Unable to load alternate character...\r\nHit enter to return to the creation process: %s", telnet_go_ahead(d));
	}
}


/**
* Sets the state and prompts.
*
* @param descriptor_data *d the user
* @param int state any CON_x
*/
void set_creation_state(descriptor_data *d, int state) {
	STATE(d) = state;
	prompt_creation(d);
}


/**
* Shows the menu of bonus traits, marking ones a player already has.
*
* @param char_data *ch The player to show the menu to.
*/
void show_bonus_trait_menu(char_data *ch) {
	int iter, count, first, second, reset, hours, advanced;
	struct time_info_data t;
	
	if (IS_NPC(ch) || !ch->desc) {
		return;
	}
	
	if (GET_ACCESS_LEVEL(ch) == 0) {
		hours = 0;
	}
	else {
		// compute playtime
		t = *real_time_passed((time(0) - ch->player.time.logon) + ch->player.time.played, 0);
		hours = t.day * 24 + t.hours;
	}
	
	// pull configs
	first = config_get_int("hours_to_first_bonus_trait");
	second = config_get_int("hours_to_second_bonus_trait");
	reset = config_get_int("hours_to_bonus_trait_reset");
	advanced = config_get_int("hours_to_advanced_bonus_traits");
	
	msg_to_char(ch, "\r\nBonus Traits (with %d hour%s of play):\r\n", hours, PLURAL(hours));
	
	// info section
	if (first > 0) {
		msg_to_char(ch, "You get one bonus trait after %d hour%s of play", first, PLURAL(first));
	}
	else if (second > 0) {
		msg_to_char(ch, "You get one bonus trait automatically");
	}
	else {
		msg_to_char(ch, "You get two bonus traits automatically");
	}
	if (second > 0) {
		msg_to_char(ch, " and another after %d hour%s%s", second, PLURAL(second), (first == 0) ? " of play" : "");
	}
	if (reset > 0) {
		msg_to_char(ch, ".\r\nYour bonus traits automatically reset after your first %d hour%s%s.\r\n", reset, PLURAL(reset), (first == 0 && second == 0) ? " of play" : "");
	}
	else {
		msg_to_char(ch, ".\r\n");
	}
	
	msg_to_char(ch, "Add a bonus trait:\r\n");
	for (iter = 0, count = 0; iter < NUM_BONUS_TRAITS; ++iter) {
		if (IS_SET(FORBIDDEN_BONUS_TRAITS, BIT(iter))) {
			continue;	// not allowed on this mud
		}
		if (hours < advanced && !IS_SET(NEWBIE_BONUS_TRAITS, BIT(iter))) {
			continue;	// not enough playtime
		}
		msg_to_char(ch, "%2d. %s%s\r\n", ++count, bonus_bit_descriptions[iter], (HAS_BONUS_TRAIT(ch, BIT(iter)) ? " &g(already chosen)&0" : ""));
	}
	
	msg_to_char(ch, "\r\nEnter a number to choose (or 'skip' to choose later) > %s", telnet_go_ahead(ch->desc));
}


/**
* Starts the character creation process with prompt
*
* @param descriptor_data *d the user
*/
void start_creation_process(descriptor_data *d) {
	set_creation_state(d, creation_data[0].state);
}


 //////////////////////////////////////////////////////////////////////////////
//// MENU INTERPRETER FUNCTIONS //////////////////////////////////////////////


/* Determine if a person is multiplaying (FALSE), true is "ok to log in" for some reason */
bool check_multiplaying(descriptor_data *d) {
	descriptor_data *c, *next_c;
	bool ok = TRUE;

	return TRUE; // prool: multiplaying is allowed
	
	if (ACCOUNT_FLAGGED(d->character, ACCT_MULTI_CHAR)) {
		return TRUE;
	}
	
	/* Check for connected players with identical hosts */
	for (c = descriptor_list; c && ok; c = next_c) {
		next_c = c->next;
		
		if (c == d || STATE(c) != CON_PLAYING || GET_IDNUM(c->character) == GET_IDNUM(d->character)) {
			continue;
		}
		
		if (!ACCOUNT_FLAGGED(d->character, ACCT_MULTI_CHAR) && GET_ACCOUNT(d->character) == GET_ACCOUNT(c->character)) {
			// account is already online: disconnect the other one (rather than bounce them)
			SEND_TO_Q("\r\nYour login has been usurped by another character from your account!\r\n", c);
			STATE(c) = (STATE(c) == CON_PLAYING) ? CON_DISCONNECT : CON_CLOSE;
			// ok = FALSE;
		}
		else if (!ACCOUNT_FLAGGED(d->character, ACCT_MULTI_IP | ACCT_MULTI_CHAR) && !ACCOUNT_FLAGGED(c->character, ACCT_MULTI_IP | ACCT_MULTI_CHAR) && !PLR_FLAGGED(d->character, PLR_IPMASK) && !strcmp(c->host, d->host)) {
			// IP is already logged in: just decline the connection
			// anonymous players are allowed through because the IP is not expected to be unique (unless same acct)
			if (GET_ACCOUNT(d->character) == GET_ACCOUNT(c->character) || !has_anonymous_host(d)) {
				ok = FALSE;
			}
		}
	}
	
	return ok;
}


// simple motd
void send_motd(descriptor_data *d) {
	int i;
	
	if (text_file_strings[TEXT_FILE_SHORT_CREDITS]) {
		SEND_TO_Q(text_file_strings[TEXT_FILE_SHORT_CREDITS], d);
	}

	SEND_TO_Q(" ", d);
	for (i = 0; i < 79; i++)
		SEND_TO_Q("=", d);
	SEND_TO_Q("\r\n\r\n", d);

	if (IS_IMMORTAL(d->character) && text_file_strings[TEXT_FILE_IMOTD]) {
		SEND_TO_Q(text_file_strings[TEXT_FILE_IMOTD], d);
	}
	else if (text_file_strings[TEXT_FILE_MOTD]) {
		SEND_TO_Q(text_file_strings[TEXT_FILE_MOTD], d);
	}

	SEND_TO_Q("\r\n ", d);
	for (i = 0; i < 79; i++)
		SEND_TO_Q("=", d);
	SEND_TO_Q("\r\n", d);
}


/**
* Sends the MOTD, MXP tags, and other data that should be shown when a player
* logs in, which may happen in several different ways.
*
* @param descriptor_data *desc The person to send it to.
* @param int bad_pws Number of bad password attempts, which sometimes must be retrieved and cleared ahead of time.
*/
void send_login_motd(descriptor_data *desc, int bad_pws) {
	char buf[MAX_STRING_LENGTH];
	send_motd(desc);
	MXPSendTag(desc, "<VERSION>");
	
	/* Check bad passwords */
	if (bad_pws) {
		sprintf(buf, "\r\n\r\n\007\007\007&r%d LOGIN FAILURE%s SINCE LAST SUCCESSFUL LOGIN.&0\r\n", bad_pws, (bad_pws > 1) ? "S" : "");
		SEND_TO_Q(buf, desc);
		GET_BAD_PWS(desc->character) = 0;
	}

	/* Check previous logon */
	if (GET_PREV_HOST(desc->character) && GET_PREV_LOGON(desc->character) > 0) {
		sprintf(buf, "Your last login was on %6.10s from %s.\r\n", ctime(&GET_PREV_LOGON(desc->character)), GET_PREV_HOST(desc->character));
		SEND_TO_Q(buf, desc);
	}
}


/*
 * XXX: Make immortals 'return' instead of being disconnected when switched
 *      into person returns.  This function seems a bit over-extended too.
 */
int perform_dupe_check(descriptor_data *d) {
	descriptor_data *k, *next_k;
	char_data *target = NULL, *ch, *next_ch;
	int mode = 0;

	#define RECON		1
	#define USURP		2
	#define UNSWITCH	3

	int id = GET_IDNUM(d->character);

	/*
	 * Now that this descriptor has successfully logged in, disconnect all
	 * other descriptors controlling a character with the same ID number.
	 */

	for (k = descriptor_list; k; k = next_k) {
		next_k = k->next;

		if (k == d)
			continue;

		if (k->original && (GET_IDNUM(k->original) == id)) {    /* switched char */
			SEND_TO_Q("\r\nMultiple login detected -- disconnecting.\r\n", k);
			STATE(k) = CON_CLOSE;
			if (!target) {
				target = k->original;
				mode = UNSWITCH;
			}
			if (k->character) {
				k->character->desc = NULL;
			}
			k->character = NULL;
			k->original = NULL;
		}
		else if (k->character && (GET_IDNUM(k->character) == id)) {
			if (!target && STATE(k) == CON_PLAYING) {
				SEND_TO_Q("\r\nThis body has been usurped!\r\n", k);
				target = k->character;
				mode = USURP;
			}
			k->character->desc = NULL;
			k->character = NULL;
			k->original = NULL;
			SEND_TO_Q("\r\nMultiple login detected -- disconnecting.\r\n", k);
			STATE(k) = CON_CLOSE;
		}
	}

	/*
	 * now, go through the character list, deleting all characters that
	 * are not already marked for deletion from the above step (i.e., in the
	 * CON_HANGUP state), and have not already been selected as a target for
	 * switching into.  In addition, if we haven't already found a target,
	 * choose one if one is available (while still deleting the other
	 * duplicates, though theoretically none should be able to exist).
	 */
	
	DL_FOREACH_SAFE2(player_character_list, ch, next_ch, next_plr) {
		if (GET_IDNUM(ch) != id)
			continue;

		/* ignore chars with descriptors (already handled by above step) */
		if (ch->desc)
			continue;

		/* don't extract the target char we've found one already */
		if (ch == target)
			continue;

		/* we don't already have a target and found a candidate for switching */
		if (!target) {
			target = ch;
			mode = RECON;
			continue;
		}

		/* we've found a duplicate - blow him away. */
		if (IN_ROOM(ch))
			char_from_room(ch);
		char_to_room(ch, world_table);	// put somewhere extractable
		extract_char(ch);
	}

	/* no target for swicthing into was found - allow login to continue */
	if (!target)
		return (0);

	/* Okay, we've found a target.  Connect d to target. */
	free_char(d->character); /* get rid of the old char */
	d->character = target;
	d->character->desc = d;
	d->original = NULL;
	GET_IDLE_SECONDS(d->character) = 0;
	REMOVE_BIT(PLR_FLAGS(d->character), PLR_MAILING);
	STATE(d) = CON_PLAYING;

	if (PLR_FLAGGED(d->character, PLR_IPMASK)) {
		strcpy(d->host, "masked");
	}
	
	// update stored host
	if (IN_ROOM(d->character)) {
		if (GET_PREV_HOST(d->character)) {
			free(GET_PREV_HOST(d->character));
		}
		GET_PREV_HOST(d->character) = strdup(d->host);
	}

	switch (mode) {
		case RECON:
			SEND_TO_Q("Reconnecting.\r\n", d);
			act("$n has reconnected.", TRUE, d->character, 0, 0, TO_ROOM);
			syslog(SYS_LOGIN, GET_INVIS_LEV(d->character), TRUE, "%s [%s] has reconnected at %s", GET_NAME(d->character), d->host, IN_ROOM(d->character) ? room_log_identifier(IN_ROOM(d->character)) : "an unknown location");
			break;
		case USURP:
			SEND_TO_Q("You take over your own body, already in use!\r\n", d);
			act("$n suddenly keels over in pain, surrounded by a white aura...\r\n"
				"$n's body has been taken over by a new spirit!", TRUE, d->character, 0, 0, TO_ROOM);
			syslog(SYS_LOGIN, GET_INVIS_LEV(d->character), TRUE, "%s [%s] has re-logged in at %s ... disconnecting old socket", GET_NAME(d->character), d->host, IN_ROOM(d->character) ? room_log_identifier(IN_ROOM(d->character)) : "an unknown location");
			break;
		case UNSWITCH:
			SEND_TO_Q("Reconnecting to unswitched char.", d);
			syslog(SYS_LOGIN, GET_INVIS_LEV(d->character), TRUE, "%s [%s] has reconnected at %s", GET_NAME(d->character), d->host, IN_ROOM(d->character) ? room_log_identifier(IN_ROOM(d->character)) : "an unknown location");
			break;
	}
	
	refresh_all_quests(d->character);
	MXPSendTag(d, "<VERSION>");
	send_initial_MSDP(d);
	
	return (1);
}


/**
* basic name validation and processing
*
* @param char *arg The name from the user.
* @param char *name A buffer to store the modified name to.
* @param descriptor_data *desc Optional: The person who will receive the error message if this fails. (May be NULL,)
* @param bool reduced_restrictions For "set" or other immortal commands: checks fewer things if TRUE.
* @return int 1 if the name was invalid (and sent a message); 0 if it was valid (no message sent).
*/
int _parse_name(char *arg, char *name, descriptor_data *desc, bool reduced_restrictions) {
	int iter, caps;
	descriptor_data *desc_iter;
	
	const char *allowed_chars = "'-";

	// skip whitespace
	skip_spaces(&arg);
	
	// validate parts of the string
	caps = 0;
	for (iter = 0; iter < strlen(arg); ++iter) {
		if (!isalpha(arg[iter]) && !strchr(allowed_chars, arg[iter])) {
			if (desc) {
				msg_to_desc(desc, "Names cannot contain '%c'.\r\n", arg[iter]);
			}
			return 1;
		}
		else if (isupper(arg[iter])) {
			++caps;
		}
	}
	if (!reduced_restrictions && config_get_int("max_capitals_in_name") > 0 && caps > config_get_int("max_capitals_in_name")) {
		if (desc) {
			msg_to_desc(desc, "Names may not contain more than %d capital letters.\r\n", config_get_int("max_capitals_in_name"));
		}
		return 1;
	}
	if (!reduced_restrictions && (strlen(arg) < 2 || strlen(arg) > MAX_NAME_LENGTH)) {
		if (desc) {
			msg_to_desc(desc, "Name must be between 2 and %d letters in length.\r\n", MAX_NAME_LENGTH);
		}
		return 1;
	}
	if (reduced_restrictions && strlen(arg) > MAX_NAME_LENGTH * 2) {
		if (desc) {
			msg_to_desc(desc, "Name too long.\r\n");
		}
		return 1;
	}
	if (!isalpha(*arg)) {
		if (desc) {
			msg_to_desc(desc, "Name must begin with a letter.\r\n");
		}
		return 1;
	}
	if (!isalpha(arg[strlen(arg) - 1])) {
		if (desc) {
			msg_to_desc(desc, "Name must end with a letter.\r\n");
		}
		return 1;
	}
	if (fill_word(arg) || reserved_word(arg)) {
		if (desc) {
			msg_to_desc(desc, "'&&Z%s' is an invalid name.\r\n", arg);
		}
		return 1;
	}
	
	// check other descriptors for the same name
	LL_FOREACH(descriptor_list, desc_iter) {
		if (desc_iter->character && GET_NAME(desc_iter->character) && !str_cmp(GET_NAME(desc_iter->character), arg)) {
			if (STATE(desc_iter) == CON_PLAYING || STATE(desc_iter) == CON_RMOTD) {
				// playing state: name is valid (maybe just reconnecting)
				continue;
			}
			else {
				if (desc) {
					msg_to_desc(desc, "Someone is already using that name.\r\n");
				}
				return 1;
			}
		}
	}
	
	// seems good
	strcpy(name, arg);
	return (0);
}


/**
* Main "socket nanny" for processing menu input.
*/
void nanny(descriptor_data *d, char *arg) {
	char tmp_name[MAX_INPUT_LENGTH];
	int load_result, i, iter, num;
	bitvector_t bit;
	bool show_start = FALSE;
	char_data *temp_char;
	
	// this avoids treating telnet negotiation as menu input
	if (d->no_nanny) {
		d->no_nanny = FALSE;
		return;
	}

	skip_spaces(&arg);
	d->idle_tics = 0;

	switch (STATE(d)) {
		case CON_GET_NAME: {	/* wait for input of name */
			// ensure no characters are loaded/pending anything before loading one here
			run_delayed_refresh();
			free_loaded_players();
			
			if (d->character == NULL) {
				CREATE(d->character, char_data, 1);
				clear_char(d->character);
				init_player_specials(d->character);
				d->character->desc = d;
				GET_PREV_HOST(d->character) = str_dup(d->host);	// this will be overwritten if it's not a new char
			}
			
			if (!*arg) {
				SET_BIT(PLR_FLAGS(d->character), PLR_KEEP_LAST_LOGIN_INFO);	// prevent login storing
				STATE(d) = CON_CLOSE;
			}
			else if (!str_cmp(arg, "new")) {
				// special case for players who typed "new"
				msg_to_desc(d, "\r\nEnter new character name: %s", telnet_go_ahead(d));
				return;
			}
			else {
				if (_parse_name(arg, tmp_name, d, FALSE)) {
					msg_to_desc(d, "Please try another name: %s", telnet_go_ahead(d));
					return;
				}
				if ((temp_char = load_player(tmp_name, TRUE))) {
					free_char(d->character);
					d->character = temp_char;	// can't load directly; overwrites the existing char
					d->character->desc = d;
					check_delayed_load(d->character);
					
					/* undo it just in case they are set */
					REMOVE_BIT(PLR_FLAGS(d->character), PLR_MAILING);

					// flush messages because updated items appear here
					send_stacked_msgs(d);
					
					msg_to_desc(d, "Password: %s", telnet_go_ahead(d));
					ProtocolNoEcho(d, true);
					STATE(d) = CON_PASSWORD;
				}
				else {
					/* player unknown -- make new character */
					const char *rules = config_get_string("name_rules");

					// Check bans and multiple creations of a character.
					if (!Valid_Name(tmp_name)) {
						msg_to_desc(d, "Invalid name, please try another.\r\nName: %s", telnet_go_ahead(d));
						return;
					}
					GET_PC_NAME(d->character) = str_dup(CAP(tmp_name));

					msg_to_desc(d, "\r\nNew character:\r\n");
					if (rules && *rules) {
						msg_to_desc(d, "%s\r\n", rules);
					}

					msg_to_desc(d, "Did I get that name right, %s (Y/N)? %s", tmp_name, telnet_go_ahead(d));
					STATE(d) = CON_NAME_CNFRM;
				}
			}
			break;
		}

		case CON_NAME_CNFRM: {	/* wait for conf. of new name    */
			if (UPPER(*arg) == 'Y') {
				if (isbanned(d->host) >= BAN_NEW) {
					syslog(SYS_LOGIN, 0, TRUE, "Request for new char %s denied from [%s] (siteban)", GET_PC_NAME(d->character), d->host);
					SEND_TO_Q("Sorry, new characters are not allowed from your site!\r\n", d);
					STATE(d) = CON_CLOSE;
					return;
				}
				if (wizlock_level) {
					if (!wizlock_message)
						SEND_TO_Q("Sorry, new players can't be created at the moment.\r\n", d);
					else {
						msg_to_desc(d, "%s\r\n", wizlock_message);
					}
					syslog(SYS_LOGIN, 0, TRUE, "Request for new char %s denied from [%s] (wizlock)", GET_PC_NAME(d->character), d->host);
					STATE(d) = CON_CLOSE;
					return;
				}
				
				start_creation_process(d);
			}
			else if (*arg == 'n' || *arg == 'N') {
				msg_to_desc(d, "Okay, what IS it, then? %s", telnet_go_ahead(d));
				free(GET_PC_NAME(d->character));
				GET_PC_NAME(d->character) = NULL;
				STATE(d) = CON_GET_NAME;
			}
			else {
				msg_to_desc(d, "Please type Yes or No: %s", telnet_go_ahead(d));
			}
			break;
		}

		case CON_PASSWORD: {	/* get pwd for known player      */
			/*
			 * To really prevent duping correctly, the player's record should
			 * be reloaded from disk at this point (after the password has been
			 * typed).  However I'm afraid that trying to load a character over
			 * an already loaded character is going to cause some problem down the
			 * road that I can't see at the moment.  So to compensate, I'm going to
			 * (1) add a 15 or 20-second time limit for entering a password, and (2)
			 * re-add the code to cut off duplicates when a player quits.  JE 6 Feb 96
			 */

			/* New echo-on eats the return on telnet. Extra space better than none. */
			SEND_TO_Q("\r\n", d);

			if (!*arg) {
				SET_BIT(PLR_FLAGS(d->character), PLR_KEEP_LAST_LOGIN_INFO);	// prevent login from storing
				STATE(d) = CON_CLOSE;
			}
			else {
				if (strncmp(CRYPT(arg, PASSWORD_SALT), GET_PASSWD(d->character), MAX_PWD_LENGTH)) {
					syslog(SYS_LOGIN, 0, TRUE, "BAD PW: %s [%s]", GET_NAME(d->character), d->host);
					if ((temp_char = is_playing(GET_IDNUM(d->character)))) {
						// update on in-game version instead
						++GET_BAD_PWS(temp_char);
						SAVE_CHAR(temp_char);
					}
					else {
						// update on this one (not in-game)
						++GET_BAD_PWS(d->character);
						SET_BIT(PLR_FLAGS(d->character), PLR_KEEP_LAST_LOGIN_INFO);
						SAVE_CHAR(d->character);
					}
					
					// 3 strikes and you're out.
					if (++(d->bad_pws) >= config_get_int("max_bad_pws")) {
						SEND_TO_Q("Wrong password... disconnecting.\r\n", d);
						STATE(d) = CON_CLOSE;
					}
					else {
						msg_to_desc(d, "Wrong password.\r\nPassword: %s", telnet_go_ahead(d));
					}
					return;
				}
				
				// echo back on
				ProtocolNoEcho(d, false);

				/* Password was correct. */
				load_result = GET_BAD_PWS(d->character);
				GET_BAD_PWS(d->character) = 0;
				d->bad_pws = 0;

				if (isbanned(d->host) == BAN_SELECT && !ACCOUNT_FLAGGED(d->character, ACCT_SITEOK)) {
					SEND_TO_Q("Sorry, this account has not been cleared for login from your site!\r\n", d);
					STATE(d) = CON_CLOSE;
					syslog(SYS_LOGIN, 0, TRUE, "Connection attempt for %s denied from %s", GET_NAME(d->character), d->host);
					return;
				}
				if (GET_ACCESS_LEVEL(d->character) < wizlock_level) {
					if (wizlock_message) {
						msg_to_desc(d, "%s\r\n", wizlock_message);
					}
					else
						SEND_TO_Q("The game is temporarily restricted... try again later.\r\n", d);
					STATE(d) = CON_CLOSE;
					syslog(SYS_LOGIN, 0, TRUE, "Request for login denied for %s [%s] (wizlock)", GET_NAME(d->character), d->host);
					return;
				}
				/* check and make sure no other copies of this player are logged in */
				if (!check_multiplaying(d)) {
					SEND_TO_Q("\r\n\033[31mAccess Denied: Multiplaying detected\033[0m\r\n", d);

					SEND_TO_Q("There is already a character logged in from the same IP address or account as\r\n", d);
					SEND_TO_Q("you. If you are controlling that character, you must remove it from the game\r\n", d);
					SEND_TO_Q("before this character can enter. Rarely, computers may share IP addresses. If\r\n", d);
					SEND_TO_Q("this is the case, we may grant exceptions. You will need to petition the staff\r\n", d);
					SEND_TO_Q("to allow more than one person from your IP if you have multiple people who want\r\n", d);
					SEND_TO_Q("to play together.\r\n", d);
					SEND_TO_Q("\r\n", d);
					SEND_TO_Q("Press ENTER to continue:\r\n", d);
					syslog(SYS_LOGIN, 0, TRUE, "Login denied: Multiplaying detected for %s [%s]", GET_NAME(d->character), d->host);

					STATE(d) = CON_GOODBYE;
					return;
				}
				
				if (perform_dupe_check(d))
					return;
				
				if (!PLR_FLAGGED(d->character, PLR_INVSTART)) {
					syslog(SYS_LOGIN, GET_INVIS_LEV(d->character), TRUE, "%s [%s] has connected", GET_NAME(d->character), PLR_FLAGGED(d->character, PLR_IPMASK) ? "masked" : d->host);
				}
				
				// check if traits need to be reset
				if (check_bonus_trait_reset(d->character)) {
					SEND_TO_Q("Press ENTER to continue:\r\n", d);
					STATE(d) = CON_BONUS_RESET;
					return;
				}
				// otherwise check here if they need more traits than they have
				else if (num_earned_bonus_traits(d->character) > count_bits(GET_BONUS_TRAITS(d->character))) {
					set_creation_state(d, CON_BONUS_TRAIT);
					return;
				}
				
				send_login_motd(d, load_result);
				
				// send on to motd
				msg_to_desc(d, "\r\n*** Press ENTER: %s", telnet_go_ahead(d));
				STATE(d) = CON_RMOTD;
			}
			break;
		}

		case CON_NEWPASSWD: {
			if (!*arg || strlen(arg) > MAX_PWD_LENGTH || strlen(arg) < 3 || !str_cmp(arg, GET_PC_NAME(d->character))) {
				SEND_TO_Q("\r\nIllegal password.\r\n", d);
				msg_to_desc(d, "Password: %s", telnet_go_ahead(d));
				return;
			}
			
			GET_PASSWD(d->character) = str_dup(CRYPT(arg, PASSWORD_SALT));
			next_creation_step(d);
			break;
		}

		case CON_CNFPASSWD: {
			if (strncmp(CRYPT(arg, PASSWORD_SALT), GET_PASSWD(d->character), MAX_PWD_LENGTH)) {
				SEND_TO_Q("\r\nPasswords don't match... start over.\r\n", d);
				msg_to_desc(d, "Password: %s", telnet_go_ahead(d));
				STATE(d) = CON_NEWPASSWD;
				return;
			}
			ProtocolNoEcho(d, false);
			next_creation_step(d);
			break;
		}

		case CON_QLAST_NAME: {	/* Want a last name? */
			if (UPPER(*arg) == 'Y') {
				set_creation_state(d, CON_SLAST_NAME);
			}
			else if (UPPER(*arg) == 'N') {
				change_personal_lastname(d->character, NULL);
				set_creation_state(d, CON_QSEX);
				break;
			}
			else {
				msg_to_desc(d, "\r\nPlease type Yes or No: %s", telnet_go_ahead(d));
			}
			
			break;
		}

		case CON_Q_SCREEN_READER: {
			if (UPPER(*arg) == 'Y') {
				SET_BIT(PRF_FLAGS(d->character), PRF_SCREEN_READER | PRF_COMPACT | PRF_SCROLLING);
				set_creation_state(d, CON_Q_HAS_ALT);
			}
			else if (UPPER(*arg) == 'N') {
				REMOVE_BIT(PRF_FLAGS(d->character), PRF_SCREEN_READER);
				next_creation_step(d);
			}
			else {
				msg_to_desc(d, "\r\nPlease type Yes or No: %s", telnet_go_ahead(d));
			}
			break;
		}

		case CON_Q_HAS_ALT: {
			if (UPPER(*arg) == 'Y') {
				next_creation_step(d);
			}
			else if (UPPER(*arg) == 'N') {
				set_creation_state(d, CON_Q_ARCHETYPE);
			}
			else {
				msg_to_desc(d, "\r\nPlease type Yes or No: %s", telnet_go_ahead(d));
			}
			break;
		}
		case CON_Q_ALT_NAME: {
			process_alt_name(d, arg);
			break;
		}
		case CON_Q_ALT_PASSWORD: {
			SEND_TO_Q("\r\n", d);	// echo-off usually hides the CR
			process_alt_password(d, arg);
			break;
		}

		case CON_SLAST_NAME: {	/* What's yer last name? */
			if (!*arg) {
				msg_to_desc(d, "\r\nEnter a last name: %s", telnet_go_ahead(d));
				return;
			}
			else if (_parse_name(arg, tmp_name, d, FALSE) || !Valid_Name(tmp_name)) {
				msg_to_desc(d, "\r\nPlease try a different last name: %s", telnet_go_ahead(d));
				return;
			}
			else {
				change_personal_lastname(d->character, tmp_name);
				if (GET_PERSONAL_LASTNAME(d->character) && !GET_CURRENT_LASTNAME(d->character)) {
					GET_CURRENT_LASTNAME(d->character) = str_dup(GET_PERSONAL_LASTNAME(d->character));
				}
				next_creation_step(d);
			}
			break;
		}

		case CON_CLAST_NAME: {	/* Wait for conf. of last name    */
			if (UPPER(*arg) == 'Y') {
				next_creation_step(d);
			}
			else if (UPPER(*arg) == 'N') {
				msg_to_desc(d, "Okay, what IS it, then? %s", telnet_go_ahead(d));
				change_personal_lastname(d->character, NULL);
				STATE(d) = CON_SLAST_NAME;
			}
			else {
				msg_to_desc(d, "Please type Yes or No: %s", telnet_go_ahead(d));
			}
			break;
		}

		case CON_QSEX: {	/* query sex of new user         */
			switch (LOWER(*arg)) {
				case 'm':
					d->character->player.sex = SEX_MALE;
					break;
				case 'f':
					d->character->player.sex = SEX_FEMALE;
					break;
				default:
					msg_to_desc(d, "That is not a sex...\r\nWhat IS your sex? %s", telnet_go_ahead(d));
					return;
			}

			next_creation_step(d);
			break;
		}
		
		case CON_REFERRAL: {
			if (*arg) {
				// store for later
				arg[MAX_REFERRED_BY_LENGTH-1] = '\0';
				if (GET_REFERRED_BY(d->character)) {
					free(GET_REFERRED_BY(d->character));
				}
				GET_REFERRED_BY(d->character) = str_dup(arg);
			}
			
			next_creation_step(d);
			break;
		}

		case CON_FINISH_CREATION: {
			// some finalization
			
			if (GET_ACCESS_LEVEL(d->character) == 0) {
				// set to base level now
				GET_ACCESS_LEVEL(d->character) = 1;
			}
			init_player(d->character);
			SAVE_CHAR(d->character);
			
			send_login_motd(d, GET_BAD_PWS(d->character));
			
			msg_to_desc(d, "\r\n*** Press ENTER: %s", telnet_go_ahead(d));
			STATE(d) = CON_RMOTD;

			syslog(SYS_LOGIN, 0, TRUE, "NEW: %s [%s] (promo: %s)", GET_NAME(d->character), d->host, GET_PROMO_ID(d->character) > 0 ? promo_codes[GET_PROMO_ID(d->character)].code : "none");
			if (GET_REFERRED_BY(d->character) && *GET_REFERRED_BY(d->character)) {
				syslog(SYS_LOGIN, 0, FALSE, "Referral: %s", GET_REFERRED_BY(d->character));
			}
			break;
		}

		case CON_Q_ARCHETYPE: {
			parse_archetype_menu(d, arg);
			break;
		}
		
		case CON_ARCHETYPE_CNFRM: {
			if (is_abbrev(arg, "yes")) {
				next_creation_step(d);
			}
			else if (is_abbrev(arg, "no")) {
				set_creation_state(d, CON_Q_ARCHETYPE);
			}
			else {
				msg_to_desc(d, "\r\nPlease type YES or NO: %s", telnet_go_ahead(d));
			}
			break;
		}
		
		case CON_PROMO_CODE: {
			int promo = -1;
			
			skip_spaces(&arg);
			if (!*arg) {
				// skip entirely
				set_creation_state(d, CON_REFERRAL);
				return;
			}
			
			for (iter = 0; *promo_codes[iter].code != '\n'; ++iter) {
				if (!promo_codes[iter].expired && !str_cmp(arg, promo_codes[iter].code)) {
					promo = iter;
					break;
				}
			}
			
			GET_PROMO_ID(d->character) = promo;
			
			// pass off based on code validity
			if (promo < 0) {
				set_creation_state(d, CON_CONFIRM_PROMO_CODE);
			}
			else {
				SEND_TO_Q("\r\nPromo code accepted.\r\n", d);
				set_creation_state(d, CON_REFERRAL);
			}
			break;
		}
		
		case CON_CONFIRM_PROMO_CODE: {
			switch (LOWER(*arg)) {
				case 'y': {
					next_creation_step(d);
					break;
				}
				case 'n': {
					set_creation_state(d, CON_PROMO_CODE);
					break;
				}
				default: {
					msg_to_desc(d, "Please type YES or NO: %s", telnet_go_ahead(d));
					return;
				}
			}
			break;
		}

		case CON_RMOTD: {		/* read CR after printing motd   */
			const char *msg;
			
			if (PLR_FLAGGED(d->character, PLR_IPMASK)) {
				strcpy(d->host, "masked");
			}
	
			// READY TO ENTER THE GAME
			if (!check_multiplaying(d)) {
				SEND_TO_Q("\r\n\033[31mAccess Denied: Multiplaying detected\033[0m\r\n", d);
				SEND_TO_Q("There is already someone logged in from the same IP address as you. If you\r\n", d);
				SEND_TO_Q("are controlling that character, you must remove it from the game before this\r\n", d);
				SEND_TO_Q("character can enter. Rarely, computers may share IP addresses. If this is\r\n", d);
				SEND_TO_Q("the case, exceptions may be granted. You will need to petition the staff\r\n", d);
				SEND_TO_Q("member in charge of authorization. When you are able to log into the mud,\r\n", d);
				SEND_TO_Q("type HELP AUTHORIZATION for the appropriate e-mail address, or contact the\r\n", d);
				SEND_TO_Q("staff member via the game.\r\n", d);
				msg_to_desc(d, "\r\nPress ENTER to continue: %s", telnet_go_ahead(d));
				syslog(SYS_LOGIN, 0, TRUE, "Login denied: Multiplaying detected for %s [%s]", GET_NAME(d->character), d->host);
				
				STATE(d) = CON_GOODBYE;
				return;
			}
			else if (IS_APPROVED(d->character) && !IS_IMMORTAL(d->character) && has_anonymous_host(d)) {
				SEND_TO_Q("\r\n\033[31mAccess Denied: Anonymous public host detected\033[0m\r\n", d);
				SEND_TO_Q("This game does not allow existing 'approved' characters to log in from public\r\n", d);
				SEND_TO_Q("hosts (such as Mudconnector) that do not provide your IP address. You can only\r\n", d);
				SEND_TO_Q("log in from this host using a character that is not 'approved', or a new\r\n", d);
				SEND_TO_Q("character (which will not be approved). To install a MUD client on your own\r\n", d);
				SEND_TO_Q("computer, visit https://empiremud.net/play-now.html\r\n", d);
				syslog(SYS_LOGIN, 0, TRUE, "Login denied: Approved character %s connecting from anonymous host [%s]", GET_NAME(d->character), d->host);
				
				STATE(d) = CON_GOODBYE;
				return;
			}
			
			if (LOWER(*arg) == 'i' && (IS_GOD(d->character) || IS_IMMORTAL(d->character))) {
				GET_INVIS_LEV(d->character) = GET_ACCESS_LEVEL(d->character);
			}
			
			// TODO most* of this block is repeated in do_alternate

			// put them in-game
			enter_player_game(d, TRUE, TRUE);
			
			msg_to_desc(d, "\r\n%s\r\n\r\n", config_get_string("welcome_message"));
			act("$n has entered the game.", TRUE, d->character, 0, 0, TO_ROOM);

			STATE(d) = CON_PLAYING;
			
			// needs newbie setup (gear, etc?)
			if (PLR_FLAGGED(d->character, PLR_NEEDS_NEWBIE_SETUP)) {
				start_new_character(d->character);
				show_start = TRUE;
			}
			
			if (AFF_FLAGGED(d->character, AFF_EARTHMELDED)) {
				msg_to_char(d->character, "You are earthmelded.\r\n");
			}
			else {
				look_at_room(d->character);
			}
			
			msg_to_char(d->character, "\r\n");	// leading \r\n between the look and the rest
			
			if (GET_LOYALTY(d->character) && EMPIRE_MOTD(GET_LOYALTY(d->character))) {
				msg_to_char(d->character, "Empire MOTD:\r\n%s\r\n", EMPIRE_MOTD(GET_LOYALTY(d->character)));
			}
			
			display_automessages_on_login(d->character);
			
			if (!PRF_FLAGGED(d->character, PRF_NO_TUTORIALS)) {
				display_tip_to_char(d->character);
			}
			
			if (GET_MAIL_PENDING(d->character)) {
				send_to_char("\trYou have mail waiting.\t0\r\n", d->character);
			}
			if ((num = has_pending_friend_requests(d->character))) {
				msg_to_char(d->character, "\tAYou have %d friend request%s waiting.\t0\r\n", num, PLURAL(num));
			}
			if (has_uncollected_event_rewards(d->character)) {
				msg_to_char(d->character, "\ttYou have uncollected event rewards. Type 'event collect' when you're in your own territory.\t0\r\n");
			}
			
			// reset daily cycle now
			check_daily_cycle_reset(d->character);
			
			if (!IS_APPROVED(d->character) && (msg = config_get_string("unapproved_greeting")) && *msg) {
				msg_to_char(d->character, "\r\n\to%s\t0", msg);
			}
			if (show_start && (msg = config_get_string("start_message")) && *msg) {
				msg_to_char(d->character, "\r\n\tY%s\t0", msg);
			}
			
			if (!IS_APPROVED(d->character) && !IS_IMMORTAL(d->character) && has_anonymous_host(d)) {
				msg_to_char(d->character, "\r\n\trWarning:\t0 You are playing from an anonymous public host, which does not provide\r\n");
				msg_to_char(d->character, "your IP address to this game. Characters from this host are not automatically\r\n");
				msg_to_char(d->character, "'approved' and only un-approved characters can play from this host. If you wish\r\n");
				msg_to_char(d->character, "to become a permanent player on this game, you should speak to an immortal who\r\n");
				msg_to_char(d->character, "can approve your character and explain how to connect from a real MUD client.\r\n\r\n");
			}
			
			d->has_prompt = 0;
			break;
		}
		
		// info page about trait reset
		case CON_BONUS_RESET: {
			// this will pass them to the right place
			set_creation_state(d, CON_BONUS_TRAIT);
			break;
		}

		// add-trait menu
		case CON_BONUS_TRAIT: {
			bool skip = FALSE;
			int hours;
			struct time_info_data t;
			
			if (!str_cmp(arg, "skip")) {
				skip = TRUE;
			}
			else {
				if (!*arg || !isdigit(*arg)) {
					// just re-show
					show_bonus_trait_menu(d->character);
					return;
				}
				
				if ((i = atoi(arg)) < 1 || i > NUM_BONUS_TRAITS) {
					msg_to_desc(d, "\r\nInvalid trait choice. Try again > %s", telnet_go_ahead(d));
					return;
				}
				
				if (GET_ACCESS_LEVEL(d->character) == 0) {
					hours = 0;
				}
				else {
					// compute playtime
					t = *real_time_passed((time(0) - d->character->player.time.logon) + d->character->player.time.played, 0);
					hours = t.day * 24 + t.hours;
				}
				
				// determine what trait they chose
				bit = 0;
				for (iter = 0; iter < NUM_BONUS_TRAITS; ++iter) {
					if (IS_SET(FORBIDDEN_BONUS_TRAITS, BIT(iter))) {
						continue;	// not allowed on this mud
					}
					if (hours < config_get_int("hours_to_advanced_bonus_traits") && !IS_SET(NEWBIE_BONUS_TRAITS, BIT(iter))) {
						continue;	// not enough playtime
					}
					
					// otherwise...
					if (--i == 0) {
						bit = BIT(iter);
						break;
					}
				}
				
				if (!bit) {
					msg_to_desc(d, "\r\nInvalid trait choice. Try again > %s", telnet_go_ahead(d));
					return;
				}
				
				if (HAS_BONUS_TRAIT(d->character, bit)) {
					msg_to_desc(d, "\r\nYou already have that trait! Try again > %s", telnet_go_ahead(d));
					return;
				}
			
				// seems ok
				SET_BIT(GET_BONUS_TRAITS(d->character), bit);
			}
			
			// only apply now if they are NOT currently doing creation -- otherwise it will be applied during creation
			if (GET_ACCESS_LEVEL(d->character) > 0) {
				if (!skip) {
					apply_bonus_trait(d->character, bit, TRUE);
					
					// didn't skip and got another trait?
					if (num_earned_bonus_traits(d->character) > count_bits(GET_BONUS_TRAITS(d->character))) {
						show_bonus_trait_menu(d->character);
						STATE(d) = CON_BONUS_TRAIT;
						return;
					}
				}
				
				// now send them to the motd
				send_login_motd(d, GET_BAD_PWS(d->character));
				
				msg_to_desc(d, "\r\n*** Press ENTER: %s", telnet_go_ahead(d));
				STATE(d) = CON_RMOTD;
			}
			else {
				// creating
				next_creation_step(d);
			}
			
			break;
		}

		case CON_GOODBYE: {	/* take an ENTER and then quit */
			SEND_TO_Q("Goodbye.\r\n", d);
			STATE(d) = CON_CLOSE;
			break;
		}

		/*
		 * It's possible, if enough pulses are missed, to kick someone off
		 * while they are at the password prompt. We'll just defer to let
		 * the game_loop() axe them.
		 */
		case CON_CLOSE: {
			break;
		}

		default: {
			log("SYSERR: Nanny: illegal state of con'ness (%d) for '%s'; closing connection.", STATE(d), d->character ? GET_NAME(d->character) : "<unknown>");
			STATE(d) = CON_DISCONNECT;	/* Safest to do. */
			break;
		}
	}
}
