#include "fight_hit.h"

#include "engine/ui/color.h"
#include "gameplay/magic/magic.h"
#include "pk.h"
#include "gameplay/statistics/dps.h"
#include "gameplay/magic/magic_utils.h"
#include "gameplay/mechanics/poison.h"
#include "gameplay/ai/mobact.h"
#include "fight.h"
#include "engine/db/global_objects.h"
#include "utils/backtrace.h"
#include "gameplay/ai/mob_memory.h"
#include "gameplay/classes/pc_classes.h"
#include "gameplay/mechanics/groups.h"
#include "gameplay/core/base_stats.h"
#include "gameplay/affects/affect_data.h"
#include "utils/utils_time.h"
#include "gameplay/mechanics/equipment.h"
#include "gameplay/skills/punctual_style.h"
#include "gameplay/skills/intercept.h"
#include "gameplay/skills/overhelm.h"
#include "gameplay/skills/mighthit.h"
#include "gameplay/skills/deviate.h"
#include "gameplay/skills/parry.h"
#include "gameplay/skills/multyparry.h"
#include "gameplay/skills/shield_block.h"
#include "gameplay/skills/backstab.h"
#include "gameplay/skills/ironwind.h"
#include "gameplay/mechanics/armor.h"
#include "gameplay/skills/addshot.h"

/**
*       .
* : 1 + ((-25)*0.4 + *0.2)/10 * /5,
*      1  2.6  
*  62.5%    37.5%   +   5 .
*        .
*/
int CalcStrCondensationDamage(CharData *ch, ObjData * /*wielded*/, int damage) {
	if (ch->IsNpc()
		|| GetRealStr(ch) <= 25
		|| !CanUseFeat(ch, EFeat::kStrengthConcentration)
		|| ch->battle_affects.get(kEafIronWind)
		|| ch->battle_affects.get(kEafOverwhelm)) {
		return damage;
	}
	double str_mod = (GetRealStr(ch) - 25.0) * 0.4;
	double lvl_mod = GetRealLevel(ch) * 0.2;
	double rmt_mod = std::min(5, GetRealRemort(ch)) / 5.0;
	double res_mod = 1.0 + (str_mod + lvl_mod) / 10.0 * rmt_mod;

	return static_cast<int>(damage * res_mod);
}

/**
*      .
* (/5 + *3) * (/(10 + /2)) * (/30)
*/
int CalcNoparryhitDmg(CharData *ch, ObjData *wielded) {
	if (!ch->GetSkill(ESkill::kNoParryHit)) return 0;

	double weap_dmg = (((GET_OBJ_VAL(wielded, 2) + 1) / 2.0) * GET_OBJ_VAL(wielded, 1));
	double weap_mod = weap_dmg / (10.0 + weap_dmg / 2.0);
	double level_mod = static_cast<double>(GetRealLevel(ch)) / 30.0;
	double skill_mod = static_cast<double>(ch->GetSkill(ESkill::kNoParryHit)) / 5.0;

	return static_cast<int>((skill_mod + GetRealRemort(ch) * 3.0) * weap_mod * level_mod);
}

// /       ( )
void GetClassWeaponMod(ECharClass class_id, const ESkill skill, int *damroll, int *hitroll) {
	int dam = *damroll;
	int calc_thaco = *hitroll;

	switch (class_id) {
		case ECharClass::kSorcerer:
			switch (skill) {
				case ESkill::kLongBlades:
					calc_thaco += 2;
					dam -= 1;
					break;
				case ESkill::kNonstandart:
					calc_thaco += 1;
					dam -= 2;
					break;
				default: break;
			}
			break;

		case ECharClass::kConjurer:
		case ECharClass::kWizard:
		case ECharClass::kCharmer: [[fallthrough]];
		case ECharClass::kNecromancer:
			switch (skill) {
				case ESkill::kAxes:
				case ESkill::kLongBlades: [[fallthrough]];
				case ESkill::kSpades:
					calc_thaco += 1;
					dam += 0;
					break;
				case ESkill::kTwohands:
					calc_thaco += 1;
					dam -= 3;
					break;
				default: break;
			}
			break;

		case ECharClass::kWarrior:
			switch (skill) {
				case ESkill::kShortBlades: [[fallthrough]];
				case ESkill::kPicks:
					calc_thaco += 2;
					dam += 0;
					break;
				default: break;
			}
			break;

		case ECharClass::kRanger:
			switch (skill) {
				case ESkill::kLongBlades: [[fallthrough]];
				case ESkill::kTwohands: calc_thaco += 1;
					dam += 0;
					break;
				default: break;
			}
			break;

		case ECharClass::kGuard:
		case ECharClass::kThief: [[fallthrough]];
		case ECharClass::kAssasine:
			switch (skill) {
				case ESkill::kAxes:
				case ESkill::kLongBlades:
				case ESkill::kTwohands: [[fallthrough]];
				case ESkill::kSpades:
					calc_thaco += 1;
					dam += 0;
					break;
				default: break;
			}
			break;
		case ECharClass::kMerchant:
			switch (skill) {
				case ESkill::kLongBlades: [[fallthrough]];
				case ESkill::kTwohands:
					calc_thaco += 1;
					dam += 0;
					break;
				default: break;
			}
			break;

		case ECharClass::kMagus:
			switch (skill) {
				case ESkill::kLongBlades:
				case ESkill::kTwohands: [[fallthrough]];
				case ESkill::kBows:
					calc_thaco += 1;
					dam += 0;
					break;
				default: break;
			}
			break;
		default: break;
	}

	*damroll = dam;
	*hitroll = calc_thaco;
}

// *    " ".
void HitData::CheckWeapFeats(const CharData *ch, ESkill weap_skill, int &calc_thaco, int &dam) {
	switch (weap_skill) {
		case ESkill::kPunch:
			if (ch->HaveFeat(EFeat::kPunchFocus)) {
				calc_thaco -= 2;
				dam += 2;
			}
			break;
		case ESkill::kClubs:
			if (ch->HaveFeat(EFeat::kClubsFocus)) {
				calc_thaco -= 2;
				dam += 2;
			}
			break;
		case ESkill::kAxes:
			if (ch->HaveFeat(EFeat::kAxesFocus)) {
				calc_thaco -= 1;
				dam += 2;
			}
			break;
		case ESkill::kLongBlades:
			if (ch->HaveFeat(EFeat::kLongsFocus)) {
				calc_thaco -= 1;
				dam += 2;
			}
			break;
		case ESkill::kShortBlades:
			if (ch->HaveFeat(EFeat::kShortsFocus)) {
				calc_thaco -= 2;
				dam += 3;
			}
			break;
		case ESkill::kNonstandart:
			if (ch->HaveFeat(EFeat::kNonstandartsFocus)) {
				calc_thaco -= 1;
				dam += 3;
			}
			break;
		case ESkill::kTwohands:
			if (ch->HaveFeat(EFeat::kTwohandsFocus)) {
				calc_thaco -= 1;
				dam += 3;
			}
			break;
		case ESkill::kPicks:
			if (ch->HaveFeat(EFeat::kPicksFocus)) {
				calc_thaco -= 2;
				dam += 3;
			}
			break;
		case ESkill::kSpades:
			if (ch->HaveFeat(EFeat::kSpadesFocus)) {
				calc_thaco -= 1;
				dam += 2;
			}
			break;
		case ESkill::kBows:
			if (ch->HaveFeat(EFeat::kBowsFocus)) {
				calc_thaco -= 7;
				dam += 4;
			}
			break;
		default: break;
	}
}

int HitData::ProcessExtradamage(CharData *ch, CharData *victim) {
	if (!check_valid_chars(ch, victim, __FILE__, __LINE__)) {
		return 0;
	}

	const int memorized_dam = dam;
	dam = std::max(0, dam);
	ProcessMighthit(ch, victim, *this);
	ProcessOverhelm(ch, victim, *this);
	ProcessPoisonedWeapom(ch, victim, *this);
	ProcessToxicMob(ch, victim, *this);
	ProcessPunctualStyle(ch, victim, *this);
	auto dmg = GenerateExtradamage(memorized_dam);
	return dmg.Process(ch, victim);
}

Damage HitData::GenerateExtradamage(int initial_dmg) {
	//   ,      .
	//  damage   
	dam = (initial_dmg >= 0 ? dam : -1);
	Damage dmg(SkillDmg(skill_num), dam, fight::kPhysDmg, wielded);
	dmg.hit_type = hit_type;
	dmg.dam_critic = dam_critic;
	dmg.flags = GetFlags();
	dmg.ch_start_pos = ch_start_pos;
	dmg.victim_start_pos = victim_start_pos;
	return dmg;
}

/**
 *      (  ), 
 *       /  .
 */
void HitData::Init(CharData *ch, CharData *victim) {
	// Find weapon for attack number weapon //
	if (weapon == fight::AttackType::kMainHand) {
		if (!(wielded = GET_EQ(ch, EEquipPos::kWield))) {
			wielded = GET_EQ(ch, EEquipPos::kBoths);
			weapon_pos = EEquipPos::kBoths;
		}
	} else if (weapon == fight::AttackType::kOffHand) {
		wielded = GET_EQ(ch, EEquipPos::kHold);
		weapon_pos = EEquipPos::kHold;
		if (!wielded) { //   
			weap_skill = ESkill::kLeftHit;
//         
//			weap_skill_is = CalcCurrentSkill(ch, weap_skill, victim);
			TrainSkill(ch, weap_skill, true, victim);
		}
	}

	if (wielded && wielded->get_type() == EObjType::kWeapon) {
		//        ,   
		weap_skill = static_cast<ESkill>(wielded->get_spec_param());
	} else {
		//   
		weap_skill = ESkill::kPunch;
	}
	if (skill_num == ESkill::kUndefined) {
		TrainSkill(ch, weap_skill, true, victim);
		SkillRollResult result = MakeSkillTest(ch, weap_skill, victim);
		weap_skill_is = result.SkillRate;
		if (result.CritLuck) {
			SendMsgToChar(ch, "   %s   .\r\n", victim->player_data.PNames[ECase::kAcc].c_str());
			act("$n $g    .", true, ch, nullptr, victim, kToVict);
			act("$n $g $N3   .", true, ch, nullptr, victim, kToNotVict);
			SetFlag(fight::kCritLuck);
			SetFlag(fight::kIgnoreSanct);
			SetFlag(fight::kHalfIgnoreArmor);
			SetFlag(fight::kIgnoreAbsorbe);
		}
	}
	//*  ESkill::kNoParryHit //
	if (skill_num == ESkill::kUndefined && ch->GetSkill(ESkill::kNoParryHit)) {
		int tmp_skill = CalcCurrentSkill(ch, ESkill::kNoParryHit, victim);
		bool success = tmp_skill >= number(1, MUD::Skill(ESkill::kNoParryHit).difficulty);
		TrainSkill(ch, ESkill::kNoParryHit, success, victim);
		if (success) {
			hit_no_parry = true;
		}
	}

	if (ch->battle_affects.get(kEafOverwhelm) || ch->battle_affects.get(kEafHammer)) {
		hit_no_parry = true;
	}

	if (wielded && wielded->get_type() == EObjType::kWeapon) {
		hit_type = GET_OBJ_VAL(wielded, 3);
	} else {
		weapon_pos = 0;
		if (ch->IsNpc()) {
			hit_type = ch->mob_specials.attack_type;
		}
	}

	//       ,    
	ch_start_pos = ch->GetPosition();
	victim_start_pos = victim->GetPosition();
}

/**
 *     ,      train_skill
 * (   weap_skill_is)   .
 * ,       ' '  - 
 * TestSelfHitroll()   .
 */
void HitData::CalcBaseHitroll(CharData *ch) {
	if (skill_num != ESkill::kThrow && skill_num != ESkill::kBackstab) {
		if (wielded
			&& wielded->get_type() == EObjType::kWeapon
			&& !ch->IsNpc()) {
			// Apply HR for light weapon
			int percent = 0;
			switch (weapon_pos) {
				case EEquipPos::kWield: percent = (str_bonus(GetRealStr(ch), STR_WIELD_W) - wielded->get_weight() + 1) / 2;
					break;
				case EEquipPos::kHold: percent = (str_bonus(GetRealStr(ch), STR_HOLD_W) - wielded->get_weight() + 1) / 2;
					break;
				case EEquipPos::kBoths:
					percent = (str_bonus(GetRealStr(ch), STR_WIELD_W) +
						str_bonus(GetRealStr(ch), STR_HOLD_W) - wielded->get_weight() + 1) / 2;
					break;
			}
			calc_thaco -= std::min(3, std::max(percent, 0));
		} else if (!ch->IsNpc()) {
			if (!CanUseFeat(ch, EFeat::kBully)) {
				calc_thaco += 4;
			} else {
				calc_thaco -= 3;
			}
		}
	}

	CheckWeapFeats(ch, weap_skill, calc_thaco, dam);

	if (ch->battle_affects.get(kEafOverwhelm) || ch->battle_affects.get(kEafHammer)) {
		calc_thaco -= std::max(0, (ch->GetSkill(weap_skill) - 70) / 8);
	}

	//    AWAKE style - decrease hitroll
	if (ch->battle_affects.get(kEafAwake)
		&& !CanUseFeat(ch, EFeat::kShadowStrike)
		&& skill_num != ESkill::kThrow
		&& skill_num != ESkill::kBackstab) {
		if (CanPerformAutoblock(ch)) {
			//          -    ( 0  10)
			calc_thaco += ch->GetSkill(ESkill::kAwake) * 5 / 100;
		} else {
			//        ,   
			//     ,      
			calc_thaco += ((ch->GetSkill(ESkill::kAwake) + 9) / 10) + 2;
		}
	}

	if (!ch->IsNpc() && skill_num != ESkill::kThrow && skill_num != ESkill::kBackstab) {
		// Casters use weather, int and wisdom
		if (IsCaster(ch)) {
			/*	  calc_thaco +=
				    (10 -
				     complex_skill_modifier (ch, kAny, GAPPLY_ESkill::SKILL_SUCCESS,
							     10));
			*/
			calc_thaco -= (int) ((GetRealInt(ch) - 13) / GetRealLevel(ch));
			calc_thaco -= (int) ((GetRealWis(ch) - 13) / GetRealLevel(ch));
		}
	}

	if (AFF_FLAGGED(ch, EAffect::kBless)) {
		calc_thaco -= 4;
	}
	if (AFF_FLAGGED(ch, EAffect::kCurse)) {
		calc_thaco += 6;
		dam -= 5;
	}

	//     
	if (ch->IsFlagged(EPrf::kPerformPowerAttack) && CanUseFeat(ch, EFeat::kPowerAttack)) {
		calc_thaco += 2;
	} else if (ch->IsFlagged(EPrf::kPerformGreatPowerAttack) && CanUseFeat(ch, EFeat::kGreatPowerAttack)) {
		calc_thaco += 4;
	} else if (ch->IsFlagged(EPrf::kPerformAimingAttack) && CanUseFeat(ch, EFeat::kAimingAttack)) {
		calc_thaco -= 2;
	} else if (ch->IsFlagged(EPrf::kPerformGreatAimingAttack) && CanUseFeat(ch, EFeat::kGreatAimingAttack)) {
		calc_thaco -= 4;
	}

	// Calculate the THAC0 of the attacker
	if (!ch->IsNpc()) {
		calc_thaco += GetThac0(ch->GetClass(), GetRealLevel(ch));
	} else {
		//     
		calc_thaco += (25 - GetRealLevel(ch) / 3);
	}

	//   
	float p_hitroll = ch->get_cond_penalty(P_HITROLL);

	calc_thaco -= static_cast<int>(GET_REAL_HR(ch) * p_hitroll);
	calc_thaco -= static_cast<int>(str_bonus(GetRealStr(ch), STR_TO_HIT) * p_hitroll);
	if ((skill_num == ESkill::kThrow
		|| skill_num == ESkill::kBackstab)
		&& wielded
		&& wielded->get_type() == EObjType::kWeapon) {
		if (skill_num == ESkill::kBackstab) {
			calc_thaco -= std::max(0, (ch->GetSkill(ESkill::kSneak) + ch->GetSkill(ESkill::kHide) - 100) / 30);
		}
	} else {
		calc_thaco += 4;
	}

	if (IsAffectedBySpell(ch, ESpell::kBerserk)) {
		if (AFF_FLAGGED(ch, EAffect::kBerserk)) {
			calc_thaco -= (12 * ((ch->get_real_max_hit() / 2) - ch->get_hit()) / ch->get_real_max_hit());
		}
	}

}

/**
 *    ,    calc_base_hr()
 * ,          
 *        CalcStaticHitroll()
 */
void HitData::CalcCircumstantialHitroll(CharData *ch, CharData *victim) {
	//   
	float p_hitroll = ch->get_cond_penalty(P_HITROLL);
	//    1   
	//  10%  "  "

	if (weapon == fight::AttackType::kOffHand
		&& skill_num != ESkill::kThrow
		&& skill_num != ESkill::kBackstab
		&& !(wielded && wielded->get_type() == EObjType::kWeapon)
		&& !ch->IsNpc()) {
			calc_thaco += std::max(0, (CalcSkillMinCap(victim, ESkill::kLeftHit) - CalcCurrentSkill(ch, ESkill::kLeftHit, victim)) / 10);
	}

	// courage
	if (IsAffectedBySpell(ch, ESpell::kCourage)) {
		int range = number(1, MUD::Skill(ESkill::kCourage).difficulty + ch->get_real_max_hit() - ch->get_hit());
		int prob = CalcCurrentSkill(ch, ESkill::kCourage, victim);
		TrainSkill(ch, ESkill::kCourage, prob > range, victim);
		if (prob > range) {
			dam += ((ch->GetSkill(ESkill::kCourage) + 19) / 20);
			calc_thaco -= static_cast<int>(((ch->GetSkill(ESkill::kCourage) + 9.0) / 20.0) * p_hitroll);
		}
	}

	// Horse modifier for attacker
	if (!ch->IsNpc() && skill_num != ESkill::kThrow && skill_num != ESkill::kBackstab && ch->IsOnHorse()) {
		TrainSkill(ch, ESkill::kRiding, true, victim);
		calc_thaco += 10 - ch->GetSkill(ESkill::kRiding) / 20;
	}

	// not can see (blind, dark, etc)
	if (!CAN_SEE(ch, victim))
		calc_thaco += (CanUseFeat(ch, EFeat::kBlindFight) ? 2 : ch->IsNpc() ? 6 : 10);
	if (!CAN_SEE(victim, ch))
		calc_thaco -= (CanUseFeat(victim, EFeat::kBlindFight) ? 2 : 8);

	// "Dirty" methods for battle
	if (skill_num != ESkill::kThrow && skill_num != ESkill::kBackstab) {
		int prob = (ch->GetSkill(weap_skill) + cha_app[GetRealCha(ch)].illusive) -
			(victim->GetSkill(weap_skill) + int_app[GetRealInt(victim)].observation);
		if (prob >= 30 && !victim->battle_affects.get(kEafAwake)
			&& (ch->IsNpc() || !ch->battle_affects.get(kEafPunctual))) {
			calc_thaco -= static_cast<int>((ch->GetSkill(weap_skill) -
				victim->GetSkill(weap_skill) > 60 ? 2 : 1) * p_hitroll);
			if (!victim->IsNpc())
				dam += (prob >= 70 ? 3 : (prob >= 50 ? 2 : 1));
		}
	}

	// AWAKE style for victim
	if (victim->battle_affects.get(kEafAwake)
		&& !AFF_FLAGGED(victim, EAffect::kStopFight)
		&& !AFF_FLAGGED(victim, EAffect::kMagicStopFight)
		&& !AFF_FLAGGED(victim, EAffect::kHold)) {
		bool success = CalcCurrentSkill(ch, ESkill::kAwake, victim)
			>= number(1, MUD::Skill(ESkill::kAwake).difficulty);
		if (success) {
			// >   ?    .
			//  . ,    
			dam -= ch->IsNpc() ? 5 : 4;
			calc_thaco += ch->IsNpc() ? 4 : 2;
		}
		TrainSkill(victim, ESkill::kAwake, success, ch);
	}

	//      
	if (weap_skill_is <= 80)
		calc_thaco -= static_cast<int>((weap_skill_is / 20.0) * p_hitroll);
	else if (weap_skill_is <= 160)
		calc_thaco -= static_cast<int>((4.0 + (weap_skill_is - 80.0) / 10.0) * p_hitroll);
	else
		calc_thaco -= static_cast<int>((12.0 + (weap_skill_is - 160.0) / 5.0) * p_hitroll);
}

// *  calc_rand_hr    '',     .
void HitData::CalcStaticHitroll(CharData *ch) {
	//   
	float p_hitroll = ch->get_cond_penalty(P_HITROLL);
	//    1   
	//  10%  "  "
	if (weapon == fight::AttackType::kOffHand
		&& skill_num != ESkill::kThrow
		&& skill_num != ESkill::kBackstab
		&& !(wielded && wielded->get_type() == EObjType::kWeapon)
		&& !ch->IsNpc()) {
		calc_thaco += (MUD::Skill(ESkill::kLeftHit).difficulty - ch->GetSkill(ESkill::kLeftHit)) / 10;
	}

	// courage
	if (IsAffectedBySpell(ch, ESpell::kCourage)) {
		dam += ((ch->GetSkill(ESkill::kCourage) + 19) / 20);
		calc_thaco -= static_cast<int>(((ch->GetSkill(ESkill::kCourage) + 9.0) / 20.0) * p_hitroll);
	}

	// Horse modifier for attacker
	if (!ch->IsNpc() && skill_num != ESkill::kThrow && skill_num != ESkill::kBackstab && ch->IsOnHorse()) {
		int prob = ch->GetSkill(ESkill::kRiding);
		int range = MUD::Skill(ESkill::kRiding).difficulty / 2;

		dam += ((prob + 19) / 10);

		if (range > prob)
			calc_thaco += ((range - prob) + 19 / 20);
		else
			calc_thaco -= ((prob - range) + 19 / 20);
	}

	//      
	if (ch->GetSkill(weap_skill) <= 80)
		calc_thaco -= static_cast<int>((ch->GetSkill(weap_skill) / 20.0) * p_hitroll);
	else if (ch->GetSkill(weap_skill) <= 160)
		calc_thaco -= static_cast<int>((4.0 + (ch->GetSkill(weap_skill) - 80.0) / 10.0) * p_hitroll);
	else
		calc_thaco -= static_cast<int>((12.0 + (ch->GetSkill(weap_skill) - 160.0) / 5.0) * p_hitroll);
}

// *    .
void HitData::CalcCircumstantialAc(CharData *victim) {
	victim_ac = CalcTotalAc(victim, victim_ac);
}

void HitData::ProcessDefensiveAbilities(CharData *ch, CharData *victim) {
	ProcessIntercept(ch, *this);
	ProcessDeviate(ch, victim, *this);
	ProcessParry(ch, victim, *this);
	ProcessMultyparry(ch, victim, *this);
	ProcessShieldBlock(ch, victim, *this);
}

/**
 *   :
 *    
 *     
 */
void HitData::AddWeaponDmg(CharData *ch, bool need_dice) {
	int damroll;
 	if (need_dice) {
		if (GetFlags()[fight::kCritLuck]) {
			damroll = GET_OBJ_VAL(wielded, 1) *  GET_OBJ_VAL(wielded, 2);
		} else {
			damroll = RollDices(GET_OBJ_VAL(wielded, 1), GET_OBJ_VAL(wielded, 2));
		}
	} else {
		damroll = (GET_OBJ_VAL(wielded, 1) * GET_OBJ_VAL(wielded, 2) + GET_OBJ_VAL(wielded, 1)) / 2;
	}
	if (ch->IsNpc()
		&& !AFF_FLAGGED(ch, EAffect::kCharmed)
		&& !(ch->IsFlagged(EMobFlag::kTutelar) || ch->IsFlagged(EMobFlag::kMentalShadow))) {
		damroll *= kMobDamageMult;
	} else {
		damroll = std::min(damroll, damroll * wielded->get_current_durability() / std::max(1, wielded->get_maximum_durability()));
	}

	damroll = CalcStrCondensationDamage(ch, wielded, damroll);
	dam += std::max(1, damroll);
}

void HitData::AddBareHandsDmg(CharData *ch, bool need_dice) {

	if (AFF_FLAGGED(ch, EAffect::kStoneHands)) {
		if (GetFlags()[fight::kCritLuck]) {
			dam += 8 + GetRealRemort(ch) / 2;
		} else {
			dam += need_dice ? RollDices(2, 4) + GetRealRemort(ch) / 2  : 5 + GetRealRemort(ch) / 2;
		}
		if (CanUseFeat(ch, EFeat::kBully)) {
			dam += GetRealLevel(ch) / 5;
			dam += std::max(0, GetRealStr(ch) - 25);
		}
	} else {
		if (GetFlags()[fight::kCritLuck]) {
			dam  += 3;
		} else {
			dam += need_dice? number(1, 3) : 2;
		}
	}
	//        ( )
	// < > <>
	// 0  50%
	// 5 100%
	// 10 150%
	// 15 200%
	//    
	if (!ch->battle_affects.get(kEafHammer)
		|| GetFlags()[fight::kCritHit]) //     
	{
		int modi = 10 * (5 + (GET_EQ(ch, EEquipPos::kHands) ? std::min(GET_EQ(ch, EEquipPos::kHands)->get_weight(), 18)
															: 0)); //   18  
		if (ch->IsNpc() || CanUseFeat(ch, EFeat::kBully)) {
			modi = std::max(100, modi);
		}
		dam = modi * dam / 100;
	}
}

// *      ( ).
void HitData::CalcCritHitChance(CharData *ch) {
	dam_critic = 0;
	int calc_critic = 0;

	// ,       //
	if ((!ch->IsNpc() && !IsMage(ch) && !IS_MAGUS(ch))
			|| (ch->IsNpc() && (!AFF_FLAGGED(ch, EAffect::kCharmed)
			&& !AFF_FLAGGED(ch, EAffect::kHelper)))) {
		calc_critic = std::min(ch->GetSkill(weap_skill), 70);
		if (CanUseFeat(ch, FindWeaponMasterFeat(weap_skill))) {
			calc_critic += std::max(0, ch->GetSkill(weap_skill) - 70);
		}
		if (CanUseFeat(ch, EFeat::kThieveStrike)) {
			calc_critic += ch->GetSkill(ESkill::kBackstab);
		}
		if (!ch->IsNpc()) {
			calc_critic += static_cast<int>(ch->GetSkill(ESkill::kPunctual) / 2);
			calc_critic += static_cast<int>(ch->GetSkill(ESkill::kNoParryHit) / 3);
		}
		calc_critic += GetRealLevel(ch) + GetRealRemort(ch);
	}
	if (number(0, 2000) < calc_critic) {
		SetFlag(fight::kCritHit);
	} else {
		ResetFlag(fight::kCritHit);
	}
}
int HitData::CalcDmg(CharData *ch, bool need_dice) {
	if (ch->IsFlagged(EPrf::kExecutor)) {
		SendMsgToChar(ch, "&Y: %s.    == %d&n\r\n", MUD::Skill(weap_skill).GetName(), dam);
	}
	if (ch->GetSkill(weap_skill) == 0) {
		calc_thaco += (50 - std::min(50, GetRealInt(ch))) / 3;
		dam -= (50 - std::min(50, GetRealInt(ch))) / 6;
	} else {
		GetClassWeaponMod(ch->GetClass(), weap_skill, &dam, &calc_thaco);
	}
	if (ch->GetSkill(weap_skill) >= 60) { //  
		dam += ((ch->GetSkill(weap_skill) - 50) / 10);
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y    == %d&n\r\n", dam);
	}
	//     
	if (ch->IsFlagged(EPrf::kPerformPowerAttack) && CanUseFeat(ch, EFeat::kPowerAttack)) {
		dam += 5;
	} else if (ch->IsFlagged(EPrf::kPerformGreatPowerAttack) && CanUseFeat(ch, EFeat::kGreatPowerAttack)) {
		dam += 10;
	} else if (ch->IsFlagged(EPrf::kPerformAimingAttack) && CanUseFeat(ch, EFeat::kAimingAttack)) {
		dam -= 5;
	} else if (ch->IsFlagged(EPrf::kPerformGreatAimingAttack) && CanUseFeat(ch, EFeat::kGreatAimingAttack)) {
		dam -= 10;
	}
	if (ch->IsFlagged(EPrf::kExecutor))
		SendMsgToChar(ch, "&Y    - == %d&n\r\n", dam);
	// courage
	if (IsAffectedBySpell(ch, ESpell::kCourage)) {
		int range = number(1, MUD::Skill(ESkill::kCourage).difficulty + ch->get_real_max_hit() - ch->get_hit());
		int prob = CalcCurrentSkill(ch, ESkill::kCourage, ch);
		if (prob > range) {
			dam += ((ch->GetSkill(ESkill::kCourage) + 19) / 20);
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y   == %d&n\r\n", dam);
		}
	}
/*	// Horse modifier for attacker
	if (!ch->IsNpc() && skill_num != ESkill::kThrow && skill_num != ESkill::kBackstab && ch->IsOnHorse()) {
		int prob = ch->get_skill(ESkill::kRiding);
		dam += ((prob + 19) / 10);
		SendMsgToChar(ch, "&Y    == %d&n\r\n", dam);
	}
*/
	//    
	if (skill_num < ESkill::kFirst) {
		dam += GetAutoattackDamroll(ch, ch->GetSkill(weap_skill));
	if (ch->IsFlagged(EPrf::kExecutor))
		SendMsgToChar(ch, "&Y +  == %d&n\r\n", dam);
	} else {
		dam += GetRealDamroll(ch);
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y + == %d&n\r\n", dam);
	}
	if (CanUseFeat(ch, EFeat::kFInesseShot) || CanUseFeat(ch, EFeat::kWeaponFinesse)) {
		dam += str_bonus(GetRealDex(ch), STR_TO_DAM);
	} else {
		dam += str_bonus(GetRealStr(ch), STR_TO_DAM);
	}
	if (ch->IsFlagged(EPrf::kExecutor))
		SendMsgToChar(ch, "&Y       == %d str_bonus == %d str == %d&n\r\n", dam, str_bonus(
						  GetRealStr(ch), STR_TO_DAM),
					  GetRealStr(ch));
	// /    ,   
	if (wielded && wielded->get_type() == EObjType::kWeapon) {
		AddWeaponDmg(ch, need_dice);
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y +   == %d  %s vnum %d&n\r\n", dam,
						  wielded->get_PName(ECase::kGen).c_str(), GET_OBJ_VNUM(wielded));
		if (GET_EQ(ch, EEquipPos::kBoths) && weap_skill != ESkill::kBows) { //   2
			dam *= 2;
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y    2 == %d&n\r\n", dam);
		}
		//  
		int tmp_dam = CalcNoparryhitDmg(ch, wielded);
		if (tmp_dam > 0) {
			// 0    = 70% ,   * 0.4 ( 5 )
			int round_dam = tmp_dam * 7 / 10;
			if (CanUseFeat(ch, EFeat::kSnakeRage)) {
				if (ch->round_counter >= 1 && ch->round_counter <= 3) {
					dam *= ch->round_counter;
				}
			}
			if (skill_num == ESkill::kBackstab || ch->round_counter <= 0) {
				dam += round_dam;
			} else {
				dam += round_dam*std::min(3, ch->round_counter);
			}
			if (ch->IsFlagged(EPrf::kExecutor))
				SendMsgToChar(ch, "&Y    == %d&n\r\n", dam);
		}
	} else {
		AddBareHandsDmg(ch, need_dice);
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y  == %d&n\r\n", dam);
	}
	if (ch->IsFlagged(EPrf::kExecutor))
		SendMsgToChar(ch, "&Y      == %d&n\r\n", dam);

	if (ch->battle_affects.get(kEafIronWind)) {
		dam += ch->GetSkill(ESkill::kIronwind) / 2;
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y     == %d&n\r\n", dam);
	}

	if (IsAffectedBySpell(ch, ESpell::kBerserk)) {
		if (AFF_FLAGGED(ch, EAffect::kBerserk)) {
			dam = (dam*std::max(150, 150 + GetRealLevel(ch) +
				RollDices(0, GetRealRemort(ch)) * 2)) / 100;
			if (ch->IsFlagged(EPrf::kExecutor))
				SendMsgToChar(ch, "&Y   == %d&n\r\n", dam);
		}
	}
	if (ch->IsNpc()) { //    
		dam += RollDices(ch->mob_specials.damnodice, ch->mob_specials.damsizedice);
	}

	if (ch->GetSkill(ESkill::kRiding) > 100 && ch->IsOnHorse()) {
		dam *= 1.0 + (ch->GetSkill(ESkill::kRiding) - 100) / 500.0;
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y    (  200 +20 )== %d&n\r\n", dam);
	}

	if (ch->add_abils.percent_physdam_add > 0) {
		int tmp;
		if (need_dice) {
			tmp = dam * (number(1, ch->add_abils.percent_physdam_add * 2) / 100.0); 
			dam += tmp;
		} else {
			tmp = dam * (ch->add_abils.percent_physdam_add / 100.0);
			dam += tmp;
		}
		if (ch->IsFlagged(EPrf::kExecutor))
			SendMsgToChar(ch, "&Y c +  == %d,  = %d  &n\r\n", dam, tmp);
	}
	//   
	dam *= ch->get_cond_penalty(P_DAMROLL);
	if (ch->IsFlagged(EPrf::kExecutor))
		SendMsgToChar(ch, "&Y    == %d&n\r\n", dam);
	return dam;

}

ESpell GetSpellIdByBreathflag(CharData *ch) {
	if (ch->IsFlagged((EMobFlag::kFireBreath))) {
		return ESpell::kFireBreath;
	} else if (ch->IsFlagged((EMobFlag::kGasBreath))) {
		return ESpell::kGasBreath;
	} else if (ch->IsFlagged((EMobFlag::kFrostBreath))) {
		return ESpell::kFrostBreath;
	} else if (ch->IsFlagged((EMobFlag::kAcidBreath))) {
		return ESpell::kAcidBreath;
	} else if (ch->IsFlagged((EMobFlag::kLightingBreath))) {
		return ESpell::kLightingBreath;
	}

	return ESpell::kUndefined;
}

/**
*   , , , , .
*/
void hit(CharData *ch, CharData *victim, ESkill type, fight::AttackType weapon) {
	if (!victim) {
		return;
	}
	if (!ch || ch->purged() || victim->purged()) {
		log("SYSERROR: ch = %s, victim = %s (%s:%d)",
			ch ? (ch->purged() ? "purged" : "true") : "false",
			victim->purged() ? "purged" : "true", __FILE__, __LINE__);
		return;
	}
	// Do some sanity checking, in case someone flees, etc.
	if (ch->in_room != victim->in_room || ch->in_room == kNowhere) {
		if (ch->GetEnemy() && ch->GetEnemy() == victim) {
			stop_fighting(ch, true);
		}
		return;
	}
	// Stand awarness mobs
	if (CAN_SEE(victim, ch)
		&& !victim->GetEnemy()
		&& ((victim->IsNpc() && (victim->get_hit() < victim->get_max_hit()
			|| victim->IsFlagged(EMobFlag::kAware)))
			|| AFF_FLAGGED(victim, EAffect::kAwarness))
		&& !AFF_FLAGGED(victim, EAffect::kHold) && victim->get_wait() <= 0) {
		set_battle_pos(victim);
	}

	//    ,  ӣ .!!
	if (type == ESkill::kUndefined) {
		ESpell spell_id;
		spell_id = GetSpellIdByBreathflag(ch);
		if (spell_id != ESpell::kUndefined) { //   
			if (!ch->GetEnemy())
				SetFighting(ch, victim);
			if (!victim->GetEnemy())
				SetFighting(victim, ch);
			// AOE   .     .
			if (ch->IsFlagged(EMobFlag::kAreaAttack)) {
				const auto
					people = world[ch->in_room]->people;    // make copy because inside loop this list might be changed.
				for (const auto &tch : people) {
					if (IS_IMMORTAL(tch) || ch->in_room == kNowhere || tch->in_room == kNowhere)
						continue;
					if (tch != ch && !group::same_group(ch, tch)) {
						CastDamage(GetRealLevel(ch), ch, tch, spell_id);
					}
				}
				return;
			}
			//    
			CastDamage(GetRealLevel(ch), ch, victim, spell_id);
			return;
		}
	}

	//go_autoassist(ch);

	//     
	HitData hit_params;
	// ,    hit()
	//c _,  _.
	hit_params.skill_num = type != ESkill::kOverwhelm && type != ESkill::kHammer ? type : ESkill::kUndefined;
	hit_params.weapon = weapon;
	hit_params.Init(ch, victim);
	//   .     . 
	if (AFF_FLAGGED(ch, EAffect::kCloudOfArrows)
		&& hit_params.skill_num == ESkill::kUndefined
		&& (ch->GetEnemy() 
		|| (!ch->battle_affects.get(kEafHammer) && !ch->battle_affects.get(kEafOverwhelm)))) {
		//     victim,  ch    
		if (ch->IsNpc()) {
			CastDamage(GetRealLevel(ch), ch, victim, ESpell::kMagicMissile);
		} else {
			CastDamage(1, ch, victim, ESpell::kMagicMissile);
		}
		if (ch->purged() || victim->purged()) { //  
			return;
		}
		if (ch->in_room != victim->in_room) {  //   
			return;
		}
		auto skillnum = GetMagicSkillId(ESpell::kCloudOfArrows);
		TrainSkill(ch, skillnum, true, victim);
	}
	//  /
	hit_params.CalcBaseHitroll(ch);
	hit_params.CalcCircumstantialHitroll(ch, victim);
	hit_params.CalcCircumstantialAc(victim);
	hit_params.CalcDmg(ch); //     
	//      
	if (hit_params.dam > 0) {
		int min_rnd = hit_params.dam - hit_params.dam / 4;
		int max_rnd = hit_params.dam + hit_params.dam / 4;
		hit_params.dam = std::max(1, number(min_rnd, max_rnd));
	}
	if (hit_params.skill_num  == ESkill::kUndefined && !hit_params.GetFlags()[fight::kCritLuck]) { //
		const int victim_lvl_miss = GetRealLevel(victim) + GetRealRemort(victim);
		const int ch_lvl_miss = GetRealLevel(ch) + GetRealRemort(ch);

		//     
		if (victim_lvl_miss - ch_lvl_miss <= 5 || (!ch->IsNpc() && !victim->IsNpc())) {
			// 5%  ,     5    
			if ((number(1, 100) <= 5)) {
				hit_params.dam = 0;
				hit_params.ProcessExtradamage(ch, victim);
				hitprcnt_mtrigger(victim);
				return;
			}
		} else {
			//   =    
			const int diff = victim_lvl_miss - ch_lvl_miss;
			if (number(1, 100) <= diff) {
				hit_params.dam = 0;
				hit_params.ProcessExtradamage(ch, victim);
				hitprcnt_mtrigger(victim);
				return;
			}
		}
		//   5%   (diceroll == 20)
		if ((hit_params.diceroll < 20 && AWAKE(victim))
			&& hit_params.calc_thaco - hit_params.diceroll > hit_params.victim_ac) {
			hit_params.dam = 0;
			hit_params.ProcessExtradamage(ch, victim);
			hitprcnt_mtrigger(victim);
			return;
		}
	}
	//   
	hit_params.CalcCritHitChance(ch);
	//   DamageEquipment,     
	if (hit_params.weapon_pos) {
		DamageEquipment(ch, hit_params.weapon_pos, hit_params.dam, 10);
	}

	if (ProcessBackstab(ch, victim, hit_params)) {
		return;
	}

	if (hit_params.skill_num == ESkill::kThrow) {
		hit_params.SetFlag(fight::kIgnoreFireShield);
		hit_params.dam *= (CalcCurrentSkill(ch, ESkill::kThrow, victim) + 10) / 10;
		if (ch->IsNpc()) {
			hit_params.dam = std::min(300, hit_params.dam);
		}
		hit_params.dam = ApplyResist(victim, EResist::kVitality, hit_params.dam);
		hit_params.ProcessExtradamage(ch, victim);
		return;
	}
	//ޣ   :
	if (!IS_CHARMICE(ch) && ch->battle_affects.get(kEafPunctual) && ch->punctual_wait <= 0 && ch->get_wait() <= 0
		&& (hit_params.diceroll >= 18 - AFF_FLAGGED(victim, EAffect::kHold))) {
		SkillRollResult result = MakeSkillTest(ch, ESkill::kPunctual, victim);
		bool success = result.success;
		TrainSkill(ch, ESkill::kPunctual, success, victim);
		if (!IS_IMMORTAL(ch)) {
			PUNCTUAL_WAIT_STATE(ch, 1 * kBattleRound);
		}
		if (success && (hit_params.calc_thaco - hit_params.diceroll < hit_params.victim_ac - 5
			|| result.CritLuck)) {
			hit_params.SetFlag(fight::kCritHit);
			hit_params.dam_critic = CalcPunctualCritDmg(ch, victim, hit_params.wielded);
			ch->send_to_TC(false, true, false, "&C   = %d&n\r\n", hit_params.dam_critic);
			victim->send_to_TC(false, true, false, "&C   = %d&n\r\n", hit_params.dam_critic);
			if (!IS_IMMORTAL(ch)) {
				PUNCTUAL_WAIT_STATE(ch, 2 * kBattleRound);
			}
			CallMagic(ch, victim, nullptr, nullptr, ESpell::kPaladineInspiration, GetRealLevel(ch));
		}
	}

	//  ,     
	if ((ch->battle_affects.get(kEafOverwhelm) || ch->battle_affects.get(kEafHammer)) && ch->get_wait() > 0) {
		ch->battle_affects.unset(kEafOverwhelm);
		ch->battle_affects.unset(kEafHammer);
	}

	//    (, , , , )
	hit_params.ProcessDefensiveAbilities(ch, victim);

	//  
	ch->send_to_TC(false, true, true, "&CΣ:   = %d&n\r\n", hit_params.dam);
	victim->send_to_TC(false, true, true, "&C:   = %d&n\r\n", hit_params.dam);
	int made_dam = hit_params.ProcessExtradamage(ch, victim);

	// ,     
	//  .      
	//  (,   ..)
/*
	if (ch->get_wait() > 0
		&& made_dam == -1
		&& (type == ESkill::kOverwhelm
			|| type == ESkill::kHammer))
	{
		ch->set_wait(0u);
	} */
	if (made_dam == -1) {
		if (type == ESkill::kOverwhelm) {
			ch->setSkillCooldown(ESkill::kOverwhelm, 0u);
		} else if (type == ESkill::kHammer) {
			ch->setSkillCooldown(ESkill::kHammer, 0u);
		}
	}

	// check if the victim has a hitprcnt trigger
	if (made_dam != -1) {
		// victim is not dead after hit
		hitprcnt_mtrigger(victim);
	}
}

//  .
void ProcessExtrahits(CharData *ch, CharData *victim, ESkill type, fight::AttackType weapon) {
	if (!ch || ch->purged()) {
		log("SYSERROR: ch = %s (%s:%d)", ch ? (ch->purged() ? "purged" : "true") : "false", __FILE__, __LINE__);
		return;
	}

	ProcessIronWindHits(ch, weapon);
	ProcessMultyShotHits(ch, victim, type, weapon);
	hit(ch, victim, type, weapon);
}

int CalcPcDamrollBonus(CharData *ch) {
	const int kMaxRemortForDamrollBonus = 35;
	const int kRemortDamrollBonus[kMaxRemortForDamrollBonus + 1] =
		{0, 0, 0, 1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8};
	int bonus = 0;
	if (IS_VIGILANT(ch) || IS_GUARD(ch) || IS_RANGER(ch)) {
		bonus = kRemortDamrollBonus[std::min(kMaxRemortForDamrollBonus, GetRealRemort(ch))];
	}
	//           ?
	// ,     ?
//	if (CanUseFeat(ch, EFeat::kBowsFocus) && ch->GetSkill(ESkill::kAddshot)) {
//		bonus *= 3;
//	}
	return bonus;
}

int CalcNpcDamrollBonus(CharData *ch) {
	if (GetRealLevel(ch) > kStrongMobLevel) {
		return static_cast<int>(GetRealLevel(ch) * number(100, 200) / 100.0);
	}
	return 0;
}

/**
*    ,     30d127
*/
int GetRealDamroll(CharData *ch) {
	if (ch->IsNpc() && !IS_CHARMICE(ch)) {
		return std::max(0, GET_DR(ch) + GET_DR_ADD(ch) + CalcNpcDamrollBonus(ch));
	}

	int bonus = CalcPcDamrollBonus(ch);
	return std::clamp(GET_DR(ch) + GET_DR_ADD(ch) + 2 * bonus, -50, (IS_MORTIFIER(ch) ? 100 : 50) + 2 * bonus);
}

int GetAutoattackDamroll(CharData *ch, int weapon_skill) {
	if (ch->IsNpc() && !IS_CHARMICE(ch)) {
		return std::max(0, GET_DR(ch) + GET_DR_ADD(ch) + CalcNpcDamrollBonus(ch));
	}
	return std::min(GET_DR(ch) + GET_DR_ADD(ch) + 2 * CalcPcDamrollBonus(ch), weapon_skill / 2); // 
}

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
